import { IFoundPageDto } from "../../Interfaces/FindAndReplace/IFoundPage";
import { SearchForm } from "../Forms/SearchForm";
import { MakeReplacementsDto } from "../Interfaces/MakeReplacementsDto";

// Type and Enum for managing sections and tab buttons
type SectionFindResultItems = {[key: string]: HTMLDivElement[]};

enum FindResultTabButtonType{
	PageBody,
	PageHead,
	Section
}

export class PageResult {
    public static ResultsCache: PageResult[] = [];

    private SearchForm: SearchForm;

    public LocalID: number;
    public OriginalResult: IFoundPageDto;
    public PageID: number;
    public PageName: string;
    public Dom: HTMLElement;
    public IsProcessing: boolean = false;
	public FindResult: IFoundPageDto;

    public constructor(findResult: IFoundPageDto, searchForm: SearchForm) {
        this.LocalID = findResult.page.id;
        this.OriginalResult = findResult;
        this.PageID = findResult.page.id;
        this.PageName = findResult.page.pageName;
        this.SearchForm = searchForm;
		this.FindResult = findResult;
        this.Dom = this.GetPageResultDOM();

        PageResult.ResultsCache.push(this); 
    }

    private GetPageResultDOM() {
		let title = `Page: ${this.PageName} (ID: ${this.PageID})`;

		// Initialize containers for different types of results.
		const headResultItems: HTMLDivElement[] = [];
		const bodyResultItems: HTMLDivElement[] = [];
		const sectionResultItems: SectionFindResultItems = {};

		// Process results for the page head.
		for (const findResult of this.FindResult.pageHeadFindResultItems){
			const stub = findResult.stub;
			const resultItem = document.createElement("div");
			const beforeSpan = document.createElement("span");
			const resultSpan = document.createElement("span");
			const afterSpan = document.createElement("span");

			resultSpan.classList.add("find-replace-search-result-result-container");

			// Use innerHTML and not textContent here so the html entities render as their display characters.
			beforeSpan.innerHTML = `${stub.beforeSanitized.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
			resultSpan.innerHTML = stub.resultSanitized;
			afterSpan.innerHTML = `${stub.afterSanitized}${stub.afterSanitized.length > 0 ? "... " : ""}`;

			resultItem.append(beforeSpan, resultSpan, afterSpan);
			headResultItems.push(resultItem);
		}

		// Handles sectioned results.
		if (this.FindResult.isSectioned){

			for (const sectionMatch of this.FindResult.sectionMatches){

				const currentSectionFindItems: HTMLDivElement[] = [];

				for (const findResult of sectionMatch.findResultItems){
					const stub = findResult.stub;
					const resultItem = document.createElement("div");
					const beforeSpan = document.createElement("span");
					const resultSpan = document.createElement("span");
					const afterSpan = document.createElement("span");
		
					resultSpan.classList.add("find-replace-search-result-result-container");
		
					// Use innerHTML and not textContent here so the html entities render as their display characters
					beforeSpan.innerHTML = `${stub.beforeSanitized.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
					resultSpan.innerHTML = stub.resultSanitized;
					afterSpan.innerHTML = `${stub.afterSanitized}${stub.afterSanitized.length > 0 ? "... " : ""}`;
		
					resultItem.append(beforeSpan, resultSpan, afterSpan);
					currentSectionFindItems.push(resultItem);
				}

				// Stores results of for this section.
				sectionResultItems[sectionMatch.sectionName] = currentSectionFindItems;
			}	
		}else{
			// Handle non-sectioned results (page body).
			for (const findResult of this.FindResult.pageBodyFindResultItems){
				const stub = findResult.stub;
				const resultItem = document.createElement("div");
				const beforeSpan = document.createElement("span");
				const resultSpan = document.createElement("span");
				const afterSpan = document.createElement("span");
	
				resultSpan.classList.add("find-replace-search-result-result-container");
	
				// Use innerHTML and not textContent here so the html entities render as their display characters
				beforeSpan.innerHTML = `${stub.beforeSanitized.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
				resultSpan.innerHTML = stub.resultSanitized;
				afterSpan.innerHTML = `${stub.afterSanitized}${stub.afterSanitized.length > 0 ? "... " : ""}`;
	
				resultItem.append(beforeSpan, resultSpan, afterSpan);
				bodyResultItems.push(resultItem);
			}
		}

		const template = document.createElement("div");
		template.classList.add("find-and-replace-find-result-item");
		template.innerHTML = `
			<div class="card">
				<div class="card-header">
					<div class="d-flex justify-content-between align-items-center">
						<h4>${title}</h4>
					</div>
				</div>
				<div class="result-card-body">
					<div class="page-tabs">
						<div class="d-flex justify-content-between align-items-center">
							<ul class="nav nav-underline page-buttons-ul">
								<li class="nav-item page-head-button-li">
									<button class="nav-link head-tab-button">Page Head</button>
								</li>
							</ul>
							<div class="edit-button">
								<a target="_blank" href="/uplift/page-editor/${this.PageID}" class="text-decoration-none">
									<span>Edit</span>
									<i class="bi bi-pencil-square"></i>
								</a>
							</div>
						</div>
					</div>
					<div class="container-fluid">
						<div class="row">
							<div class="col-12 stubs-container">
								<div class="stub-container head-result-stub-container" style="display: none;"></div>
							</div>
						</div>
					</div>
				</div>
			</div>
		`;

		const headResultStubContainer: HTMLDivElement = template.querySelector(".head-result-stub-container");
		const headTabLiElement: HTMLLIElement = template.querySelector(".page-head-button-li");
		const headTabButton: HTMLLIElement = template.querySelector(".head-tab-button");

		// Create a container for replacement buttons
		const sectionButtonContainer = document.createElement("div");
		sectionButtonContainer.classList.add("replace-section-buttons");

		let replaceHeadButton: HTMLButtonElement | undefined = undefined;
		let replaceBodyButton: HTMLButtonElement | undefined = undefined;

		// Add a replace head button if needed
		if (this.OriginalResult.pageHeadMatched){
			replaceHeadButton = document.createElement("button");
			replaceHeadButton.style.display = "none";
			replaceHeadButton.innerHTML = `
				<span>Replace in Page Head</span>
				<i class="bi bi-arrow-repeat"></i>
			`;
			replaceHeadButton.classList.add("replace-btns", "replace-page-head-btn");
			replaceHeadButton.addEventListener("click", () => this.ReplaceSectionOccurrence(FindResultTabButtonType.PageHead));
			sectionButtonContainer.appendChild(replaceHeadButton);
		}else{
			// Page head did not match. Remove the elements related to the page head
			headResultStubContainer.remove();
			headTabLiElement.remove();
		}

		// Add a replace body button if needed
		if (!this.OriginalResult.isSectioned && this.OriginalResult.pageBodyMatched){
			replaceBodyButton = document.createElement("button");
			replaceBodyButton.innerHTML = `
				<span>Replace in Page Body</span>
				<i class="bi bi-arrow-repeat"></i>
			`;
			replaceBodyButton.classList.add("replace-btns", "replace-page-body-btn");
			replaceBodyButton.addEventListener("click", () => this.ReplaceSectionOccurrence(FindResultTabButtonType.PageBody));
			sectionButtonContainer.appendChild(replaceBodyButton);
		}

		// Create a map that keeps track of which button is related to which section
		const sectionReplaceButtons: Map<string, HTMLButtonElement> = new Map();

        if (this.OriginalResult.isSectioned) {
            this.OriginalResult.sectionMatches.forEach(section => {
                const replaceSectionButton = document.createElement("button");
				replaceSectionButton.innerHTML = `
					<span>Replace in ${section.sectionName}</span>
					<i class="bi bi-arrow-repeat"></i>
				`;
				replaceSectionButton.classList.add("replace-btns", "replace-custom-section-btn");
				replaceSectionButton.setAttribute("data-section-name", section.sectionName);
				replaceSectionButton.addEventListener("click", () =>
                    this.ReplaceSectionOccurrence(FindResultTabButtonType.Section, section.sectionName)
                );
                sectionButtonContainer.appendChild(replaceSectionButton);

				// Assign sectionName and button to map
				sectionReplaceButtons.set(section.sectionName, replaceSectionButton);
            });
        }

		template.querySelector<HTMLDivElement>(".result-card-body").appendChild(sectionButtonContainer);

		// Handle tabs and adds/displays appropriate results
		if(this.FindResult.isSectioned){
			const sectionNames = Object.keys(sectionResultItems);
			if (sectionNames.length > 0) {
				for (const [index, sectionName] of sectionNames.entries()) {
					const listItem = document.createElement("li");
					listItem.classList.add("nav-item");

					const sectionMatch = this.OriginalResult.sectionMatches.find(
						match => match.sectionName === sectionName
					);

					const matchCount = sectionMatch ? sectionMatch.findResultItems.length : 0;

					const sectionButton = document.createElement("button");
					sectionButton.type = "button";

					//Appends count to custom section name
					sectionButton.textContent = `${sectionName} (${matchCount})`;
					sectionButton.classList.add("nav-link");
					sectionButton.setAttribute("data-section-name", sectionName);
					if (index === 0) {
						sectionButton.classList.add("active");
					}

					listItem.append(sectionButton);
					template.querySelector(".page-buttons-ul").insertBefore(listItem, template.querySelector(".page-head-button-li"));

					sectionButton.addEventListener("click", () => {
						this.OnFindResultTabButtonClicked(FindResultTabButtonType.Section, sectionName);
					});

					const stubContainer = document.createElement("div");
					stubContainer.classList.add("stub-container");
					stubContainer.setAttribute("data-section-name", sectionName);
					template.querySelector(".stubs-container").append(stubContainer);
					if (index !== 0) {
						stubContainer.style.display = "none";
					}

					for (const element of sectionResultItems[sectionName]) {
						stubContainer.append(element);
					}
				}
			}else{
				// Only a Page Head
				// There has to be one if this component even exists as a section component
				// but no section matches - so a page head match exists
				if (replaceHeadButton !== undefined){
					headTabButton.classList.add("active");
					replaceHeadButton.style.display = null;
					headResultStubContainer.style.display = null;
				}
			}
		}else{
			if (this.OriginalResult.pageBodyMatched) {
				const listItem = document.createElement("li");
				listItem.classList.add("nav-item");

				template.querySelector(".page-buttons-ul").append(listItem);
				const pageBodyButton = document.createElement("button");
				pageBodyButton.type = "button";
				pageBodyButton.textContent = `Page Body (${this.OriginalResult.pageBodyFindResultItems.length})`;
				pageBodyButton.classList.add("body-tab-button", "active");

				listItem.append(pageBodyButton);
				template.querySelector(".page-buttons-ul").prepend(listItem);

				pageBodyButton.addEventListener("click", () => {
					this.OnFindResultTabButtonClicked(FindResultTabButtonType.PageBody);
				});

				const stubContainer = document.createElement("div");
				stubContainer.classList.add("stub-container");
				stubContainer.classList.add("body-stub-container");
				template.querySelector(".stubs-container").append(stubContainer);

				for (const element of bodyResultItems) {
					stubContainer.append(element);
				}
			}else{
				// Page head must have matched
				if (replaceHeadButton !== undefined){
					headTabButton.classList.add("active");
					replaceHeadButton.style.display = null;
					headResultStubContainer.style.display = null;
				}
			}
		}

		// Appends count to Page Head tab
		const pageHeadMatchCount = this.OriginalResult.pageHeadFindResultItems.length;
		if (headTabButton) {
			headTabButton.textContent = `Page Head (${pageHeadMatchCount})`;

			headTabButton.addEventListener("click", () =>{
				this.OnFindResultTabButtonClicked(FindResultTabButtonType.PageHead);
			});

			for (const element of headResultItems){
				headResultStubContainer.append(element);
			}
		}

		if (this.OriginalResult.pageBodyMatched){
			const bodyStubContainer = template.querySelector<HTMLDivElement>(".body-stub-container");
			for (const element of bodyResultItems) {
				bodyStubContainer.append(element);
			}
		}

		return template;
    }

	/**
	 * When a find result tab button is clicked. Switches the tab to another one.
	 * @param buttonType
	 * @param sectionName
	 * @constructor
	 * @private
	 */
	private OnFindResultTabButtonClicked(
		buttonType: FindResultTabButtonType,
		sectionName: string = undefined
	): void{

		// Hide all replace buttons
		const buttons = this.Dom.querySelectorAll<HTMLButtonElement>(".replace-btns");
		buttons.forEach(button => button.style.display = "none");
	
		// Hide all stub containers
		const allStubContainers = this.Dom.querySelectorAll<HTMLElement>(".stub-container");
		allStubContainers.forEach(container => container.style.display = "none");

		// Removes active class from tab buttons
		this.Dom.querySelectorAll<HTMLElement>(".page-buttons-ul .nav-item")
			.forEach(listElement => listElement.querySelector("button").classList.remove("active"));

		// Handles Page Body section/tabs. Find body stub container and makes it visible,
		// adds active class to tab, and finds/displays button for replacement
		if (buttonType === FindResultTabButtonType.PageBody) {
			const bodyStubContainer = this.Dom.querySelector<HTMLElement>(".body-stub-container");
			if (bodyStubContainer) {
				bodyStubContainer.style.display = "";
			}

			const bodyTabButton = this.Dom.querySelector(".body-tab-button");
			if (bodyTabButton) {
				bodyTabButton.classList.add("active");
			}

			// Show the specific "Replace in Page Body" button
			const buttons = this.Dom.querySelectorAll<HTMLButtonElement>(".replace-btns");
			buttons.forEach(button => {
				if (button.innerHTML === `Replace in Page Body <i class="bi bi-arrow-repeat"></i>`) {
					button.style.display = "";
				}
			});
			
			// Same process for Page Body but for Page Head
		} else if (buttonType === FindResultTabButtonType.PageHead) {
			const headStubContainer = this.Dom.querySelector<HTMLElement>(".head-result-stub-container");
			if (headStubContainer) {
				headStubContainer.style.display = "";
			}

			const headTabButton = this.Dom.querySelector(".head-tab-button");
			if (headTabButton) {
				headTabButton.classList.add("active");
			}

			// Show the specific "Replace in Page Head" button
			const replacePageHeadButton = this.Dom.querySelector<HTMLButtonElement>(".replace-page-head-btn");
			replacePageHeadButton.style.display = null;

		} else if (buttonType === FindResultTabButtonType.Section && sectionName) {
			// Same process for both Page head and body but now for custom sections
			const sectionStubContainer = this.Dom.querySelector<HTMLElement>(`.stub-container[data-section-name="${sectionName}"]`);
			if (sectionStubContainer) {
				sectionStubContainer.style.display = "";
			}

			const sectionTabButton = this.Dom.querySelector(`.nav-item button[data-section-name="${sectionName}"]`);
			if (sectionTabButton) {
				sectionTabButton.classList.add("active");
			}

			// Show the specific section button
			const replaceButton = this.Dom.querySelector<HTMLButtonElement>(`.replace-btns[data-section-name="${sectionName}"]`);
			replaceButton.style.display = null;
		}
	}

	// Clear all results.
	public static ClearAll(): void {
        for(const result of PageResult.ResultsCache){
            result.Dom.remove();
        }
        PageResult.ResultsCache = [];
    }

	// Render result into a container.
	public RenderInto(container: HTMLElement): void{
		container.append(this.Dom);
	}

	// Removes this result from the DOM and cache.
    public Remove(): void {
        this.Dom.remove();
        const index = PageResult.ResultsCache.findIndex(result => result.LocalID === this.LocalID);
        if (index !== -1) {
            PageResult.ResultsCache.splice(index, 1);
        }
    }

	/**
	 * Handles the "Replace" functionality for custom sections, Page Head, and Page Body.
	 * @param buttonType 
	 * @param sectionName 
	 * @returns 
	 */
	private async ReplaceSectionOccurrence(
		buttonType: FindResultTabButtonType,
		sectionName: string = undefined
	): Promise<void> {
		if (this.IsProcessing) return;

		if (this.SearchForm.SearchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find Operation. Re-run the Find Operation before making replacements.");
			return;
		}

		this.IsProcessing = true;

		// Gets replacement text entered by the user
		const replacementInput = document.querySelector("#replacement-input");
		const replacementText = replacementInput instanceof HTMLInputElement ? replacementInput.value : "";

		// Create Clone of the original result to replace specific sections
		const clonedOriginalResult = structuredClone(this.OriginalResult);

		// Let variable to set the banner text to match section names
		let replacementSpecificName = "";

		// Let variable to target specific replacement buttons
		let buttonSelector = "";
		
		if (buttonType === FindResultTabButtonType.PageHead) {
			// Filters only matches for Page Head
			clonedOriginalResult.pageBodyFindResultItems = [];
			clonedOriginalResult.sectionMatches = [];
			replacementSpecificName = "Page Head";
		} else if (buttonType === FindResultTabButtonType.PageBody){
			// Filters only matches for Page Body
			clonedOriginalResult.pageHeadFindResultItems = [];
			clonedOriginalResult.sectionMatches = [];
			replacementSpecificName = "Page Body";
		} else if (buttonType === FindResultTabButtonType.Section && sectionName) {

			// Filters only matches for sections
			clonedOriginalResult.pageBodyFindResultItems = [];
			clonedOriginalResult.pageHeadFindResultItems = [];
			clonedOriginalResult.sectionMatches = clonedOriginalResult.sectionMatches.filter(section => section.sectionName === sectionName);
			replacementSpecificName = `Section: ${sectionName}`;
		}

		this.DisplayReplacementBanner(replacementSpecificName);

		const payload: MakeReplacementsDto = {
			findAndReplaceFoundDto: {
				foundPagesMatches: {
					foundPages: [clonedOriginalResult]
				},
				foundFiles: []
			},
			originalQuery: this.SearchForm.FindAndReplaceEntryInstance.LastQueryUsed,
			replacement: replacementText,
		};

		// If only one tab exists, remove the entire page result component when replacing
		const totalTabs = this.Dom.querySelectorAll('.page-buttons-ul .nav-item').length;
		if (totalTabs <= 1) {
			this.Remove();
		} else {
			// Remove the replaced section tab button, stub container, and replacement button
			let sectionTabLi: HTMLElement;
			let stubContainer: HTMLElement;
			let replacementButton: HTMLButtonElement;
			if (buttonType === FindResultTabButtonType.Section){
				const sectionTabButton: HTMLButtonElement = this.Dom.querySelector(`.nav-link[data-section-name="${sectionName}"]`);
				sectionTabLi = sectionTabButton.parentElement;
				stubContainer = this.Dom.querySelector(`.stub-container[data-section-name="${sectionName}"]`);
				replacementButton = this.Dom.querySelector(`.replace-btns[data-section-name="${sectionName}"]`);
			} else if (buttonType === FindResultTabButtonType.PageBody){
				const sectionTabButton: HTMLButtonElement = this.Dom.querySelector(`.body-tab-button`);
				sectionTabLi = sectionTabButton.parentElement;
				stubContainer = this.Dom.querySelector(`.body-stub-container`);
				replacementButton = this.Dom.querySelector(`.replace-page-body-btn`);
			} else if (buttonType === FindResultTabButtonType.PageHead){
				const sectionTabButton: HTMLButtonElement = this.Dom.querySelector(`.head-tab-button`);
				sectionTabLi = sectionTabButton.parentElement;
				stubContainer = this.Dom.querySelector(`.head-result-stub-container`);
				replacementButton = this.Dom.querySelector(`.replace-page-head-btn`);
			}

			sectionTabLi.remove();
			stubContainer.remove();
			replacementButton.remove();
		}
	
		const response = await fetch(`/uplift/find-and-replace/make-replacements`, {
			body: JSON.stringify(payload),
			method: "PATCH",
			headers: { "Content-Type": "application/json" },
		});
	
		let data;
		try {
			data = await response.json();
		} catch (error) {
			alert("The server responded with invalid JSON.");
			this.IsProcessing = false;
			return;
		}

		// Handles server response
		if (data.status === 1) {

		} else if (data.status === -1) {
			alert(data.error);
		}
	
		this.IsProcessing = false;
	}

	/**
	 * Adds a replacement text banner to indicate to the user that changes were made
	 * @param specificName 
	 */
	private DisplayReplacementBanner(specificName: string): void{
		const container = document.querySelector(".replacement-banner-message-container");
		const banner = document.createElement("div");
		banner.innerHTML = `Replacements were made in ${specificName}`;
		banner.classList.add("replacement-banner-message");

		if(container.childNodes.length === 3) {
			container.lastElementChild.remove();
		}

		if(container) {
			container.prepend(banner); 
		}
	
		setTimeout(() => banner.remove(), 2500);
	}
}
