export default class NoCityKeyword {
    public projectDescriptionTextField: HTMLTextAreaElement = document.querySelector("#project-description-input");
    public alertBox: HTMLElement = document.querySelector("#no-city-keyword-warning");
    public cityInput: HTMLInputElement = document.querySelector("#project-location-city-input");
    public cityPageSelector: HTMLSelectElement = document.querySelector("#city-page-id-input");

    constructor() {
        this.projectDescriptionTextField.addEventListener("input", () => {
            this.onProjectDescriptionInput();
        });

        this.cityInput.addEventListener("input", () => {
            this.onCityInputChanged();
        });

        this.cityPageSelector.addEventListener("change", () => {
            this.onCityInputChanged();
        });
    }

    onProjectDescriptionInput(){
        const city: string = this.getCity().trim();
        if (city.length > 0){
            // Find this key phrase in the content
            if (this.doesProjectDescriptionIncludeText(city)){
                // We're good
                this.hideWarning();
            }else{
                // Warn them
                this.showWarning();
            }
        }else{
            this.hideWarning();
        }
    }

    onCityInputChanged(){
        this.onProjectDescriptionInput();
    }

    /**
     * Fetches the value from the city input field or the city select menu
     */
    getCity(): string{
        if (this.cityPageSelector.value === "-1") {
            return this.cityInput.value;
        }else if (parseInt(this.cityPageSelector.value) > 0){
            // City page ID
            const cityPageName = this.cityPageSelector.options[this.cityPageSelector.selectedIndex].textContent;
            return this.getCityFromCityPageName(cityPageName);
        }

        return "";
    }

    doesProjectDescriptionIncludeText(textToCheckFor: string){
        return this.projectDescriptionTextField.value.toLowerCase().includes(textToCheckFor.toLowerCase());
    }

    showWarning(){
        this.alertBox.style.display = null;
    }

    hideWarning(){
        this.alertBox.style.display = "none";
    }

    /**
     * City pages usually have the format "City, ST" and we only want the City.
     */
    private getCityFromCityPageName(cityPageName: string): string{
        if (cityPageName.includes(",")){
            return cityPageName.split(",")[0];
        }else{
            return cityPageName;
        }
    }
}