class Paginator{

	#onPageManuallyEnteredCallbacks = [];
	#onPageNextCallbacks = [];
	#onPagePrevCallbacks = [];

	/**
	 *
	 * @param {HTMLElement} paginatorElement
	 */
	constructor(paginatorElement){
		this.currentPage = 1;
		this.maxPages = 1;
		this.paginatorElement = paginatorElement;
		this.pageForm = this.paginatorElement.querySelector("form");
		this.pageInput = this.paginatorElement.querySelector(`form input[type="number"]`);
		this.maxPagesElement = this.paginatorElement.querySelector(".paginator-max-pages");
		this.nextButton = this.paginatorElement.querySelector(".next-button");
		this.prevButton = this.paginatorElement.querySelector(".prev-button");

		this.pageForm.addEventListener("submit", e => {
			e.preventDefault();
			this.#onPageFormSubmit();
		});

		this.nextButton.addEventListener("click", () => {
			this.#onNextButtonClicked();
		});

		this.prevButton.addEventListener("click", () => {
			this.#onPrevButtonClicked();
		});
	}

	setCurrentPage(currentPage){
		this.currentPage = currentPage;
		this.pageInput.value = String(currentPage);
	}

	setMaxPages(maxPages){
		this.maxPages = maxPages;
		this.maxPagesElement.textContent = String(maxPages);
	}

	onPageManuallyEntered(callback){
		this.#onPageManuallyEnteredCallbacks.push(callback);
	}

	onPageNext(callback){
		this.#onPageNextCallbacks.push(callback);
	}

	onPagePrev(callback){
		this.#onPagePrevCallbacks.push(callback);
	}

	/**
	 * Parses the page input as an integer and, if it's a valid integer, fires all the connected event listeners.
	 */
	#onPageFormSubmit(){
		const newPage = parseInt(this.pageInput.value);
		if (!isNaN(newPage)){
			if (newPage > 0 && newPage <= this.maxPages) {
				this.setCurrentPage(newPage);
				for (const callback of this.#onPageManuallyEnteredCallbacks) {
					callback(newPage);
				}
			}
		}
	}

	#onNextButtonClicked(){
		const possibleNextPageNumber = this.currentPage + 1;
		if (possibleNextPageNumber <= this.maxPages){
			this.setCurrentPage(possibleNextPageNumber);
			for (const callback of this.#onPageNextCallbacks){
				callback(possibleNextPageNumber);
			}
		}
	}

	#onPrevButtonClicked(){
		const possiblePrevPageNumber = this.currentPage - 1;
		if (possiblePrevPageNumber > 0){
			this.setCurrentPage(possiblePrevPageNumber);
			for (const callback of this.#onPagePrevCallbacks){
				callback(possiblePrevPageNumber);
			}
		}
	}
}

export default Paginator;