class ContextMenu{

	/**
	 * @type {ContextMenu}
	 */
	static currentContextMenu = null;

	/** @type {HTMLElement} */
	dom;

	/**
	* Constructs a new context menu to be rendered at the provided
	* coordinates
	* @param {int} x
	* @param {int} y
	*/
	constructor(x,y){
		this.x = x;
		this.y = y;
		this.buildDOM();

		// Close the existing one
		if (ContextMenu.currentContextMenu !== null){
			ContextMenu.currentContextMenu.cleanup();
		}

		ContextMenu.currentContextMenu = this;
	}

	/**
	* Gets the DOM
	*/
	buildDOM(){
		this.dom = document.createElement("div");
		this.dom.classList.add("pseudo-context-menu");

		this.dom.style.left = `${this.x}px`;
		this.dom.style.top = `${this.y}px`;

		window.addEventListener("click", e => {
			this.onWindowClick(e);
		});

		window.addEventListener("contextmenu", e => {
			this.onWindowContextMenu(e);
		});
	}

	/**
	* Check if a context menu was requested outside of the current one
	*/
	onWindowContextMenu(e){
		if (e.target.closest(".pseudo-context-menu") !== this.dom){
			this.cleanup();
		}
	}

	/**
	* Checks for clicks outside of the context menu
	*/
	onWindowClick(e){
		if (e.target.closest(".pseudo-context-menu") !== this.dom){
			this.cleanup();
		}
	}

	/**
	* Adds a button with a click callback.
	* @param {string} html
	* @returns {HTMLButtonElement} The button
	*/
	addButton(html){
		const button = document.createElement("button");
		button.classList.add("cm-button");
		button.innerHTML = html;

		this.dom.append(button);

		return button
	}

	/**
	* Adds a separator element to the menu
	* @returns {HTMLDivElement} The div
	*/
	addSeparator(){
		const separator = document.createElement("div");
		separator.classList.add("cm-separator");

		this.dom.append(separator);

		return separator;
	}

	/**
	* Renders the DOM
	*/
	render(){
		document.body.append(this.dom);

		// Determine if this context menu is overflowing on the Y axis or X axis
		setTimeout(() => {

			// Y axis check
			const maxViewportYPosition = document.documentElement.clientHeight;
			const cmHeight = this.dom.clientHeight;
			const bottomYPosition = this.y + cmHeight;
			if (bottomYPosition > maxViewportYPosition){
				this.y -= cmHeight;
				this.dom.style.top = `${this.y}px`;
			}

			// X axis check
			const maxViewportXPosition = document.documentElement.clientWidth;
			const cmWidth = this.dom.clientWidth;
			const rightXPosition = this.x + cmWidth;
			if (rightXPosition > maxViewportXPosition){
				this.x -= cmWidth;
				this.dom.style.left = `${this.x}px`;
			}
		}, 1);

	}

	/**
	* Cleans up the DOM and object
	*/
	cleanup(){
		this.dom.remove();
		ContextMenu.currentContextMenu = null;
	}
}

export default ContextMenu
