import Endpoints from "../Endpoints.js";
import NewUserModal from "./modals/NewUserModal.js";
import NewUserForm from "./forms/NewUserForm.js";
import SearchUsersForm from "./forms/SearchUsersForm.js";
import Account from "./components/Account.js";
import Scheduler from "../utils/Scheduler.js";
import Paginator from "../utils/Paginator.js";

class UserAccounts {

	currentPage = 1;
	currentLimit = 25;
	currentQuery = "";
	loader = document.querySelector("#users-loader");
	isProcessing = false;

	/**
	 * @type {Paginator[]}
	 */
	paginators = [];

	constructor(){

		for (const paginator of document.querySelectorAll("pagination")){
			const newPaginator = new Paginator(paginator);
			this.hookPaginatorEvents(newPaginator);
			this.paginators.push(newPaginator);
		}

		this.loadAccounts(
			this.currentPage,
			this.currentLimit,
			this.currentQuery
		);
	}

	/**
	 *
	 * @param {Paginator} paginator
	 */
	hookPaginatorEvents(paginator){
		paginator.onPagePrev(pageNumber => {
			this.currentPage = pageNumber;
			this.loadAccounts(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});

		paginator.onPageNext(pageNumber => {
			this.currentPage = pageNumber;
			this.loadAccounts(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});

		paginator.onPageManuallyEntered(pageNumber => {
			this.currentPage = pageNumber;
			this.loadAccounts(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});
	}

	/**
	 * @param {int} page
	 * @param {int} limit
	 * @param {string} query
	 * @returns {Promise<void>}
	 */
	async loadAccounts(page, limit, query){

		if (this.isProcessing){
			return;
		}

		Account.accountsContainer.innerHTML = "";

		this.isProcessing = true;
		this.loader.style.display = "block";
		this.currentPage = page;
		this.currentLimit = limit;
		this.currentQuery = query;

		const urlParams = new URLSearchParams();
		urlParams.set("limit", String(limit));
		urlParams.set("page", String(page));
		urlParams.set("query", query);

		const response = await fetch(`${Endpoints.userAccounts.getAccounts}?${urlParams.toString()}`, {
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		await Scheduler.wait(150);

		let data;
		try{
		    /** @type {{status: int, error: ?string, accounts: Object[], totalRedirects: int, totalPages: int}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		    return;
		}

		this.loader.style.display = "none";

		if (data.status === 1){

			// Set paginator stuff
			for (const paginator of this.paginators){
				// paginator.setCurrentPage(this.currentPage);
				paginator.setMaxPages(data.totalPages);
			}

			for (/** @type {{statusCode: int, id: int, username: string, firstName: string, lastName: string, role: string, lastActivity: ?int}} */ const account of data.accounts){
				new Account(
					account.id,
					account.username,
					account.firstName,
					account.lastName,
					account.role,
					account.lastActivity
				);
			}
		}else if (data.status === -1){
			alert(data.error);
		}

		this.isProcessing = false;
	}
}

export default new UserAccounts();