class VirtualParent{

	static nextID = 1;
	/** @type {VirtualParent[]} */
	static cache = [];
	static container = document.querySelector("#virtual-parents-container");
	static #onNewVirtualParentCallbacks = [];

	/**
	 * @param {int} id
	 * @returns VirtualParent | null
	 */
	static getByID(id){
		for (const component of VirtualParent.cache){
			if (component.id === id){
				return component;
			}
		}

		return null;
	}


	/**
	 * @param {int} id
	 */
	static removeByID(id){
		for (const [index,component] of VirtualParent.cache.entries()) {
			if (component.id === id) {
				for (const option of component.linkedOptions) {
					option.remove();
				}
				component.dom.remove();
				VirtualParent.cache.splice(index, 1);
				break;
			}
		}
	}

	static onNewVirtualParent(callback){
		VirtualParent.#onNewVirtualParentCallbacks.push(callback);
	}

	static fireNewVirtualParent(virtualComponent){
		for (const callback of VirtualParent.#onNewVirtualParentCallbacks){
			callback(virtualComponent);
		}
	}

	id;
	name;

	/** @type {null|int} */
	parentID = null;
	uri = "";

	/** @type {HTMLOptionElement[]} */
	linkedOptions = [];

	constructor(name, parentID) {

		parentID = parseInt(parentID);
		if (parentID === -1){
			parentID = null;
		}

		this.id = this.constructor.nextID++;
		this.parentID = parentID;
		this.name = name;
		this.dom = this.getDOM();

		this.constructor.cache.push(this);
		this.constructor.fireNewVirtualParent(this);
	}

	getDOM(){
		const template = document.createElement("div");

		template.classList.add("virtual-parent-component");
		template.innerHTML = `
			<div class="virtual-parent-column-widths">
				<div class="id-column">
					<span>${this.id}</span>
				</div>
				<div class="name-column">
					<input type="text" class="form-control form-control-sm virtual-parent-name-input" value="${this.name}">
				</div>
				<div class="uri-column">
					<input type="text" class="form-control form-control-sm virtual-parent-uri-input" value="${this.uri}">
				</div>
				<div class="parent-name-column">
					<select class="form-control form-control-sm virtual-parent-parent-selector">
						<option value="-1">- None -</option>
					</select>
				</div>
				<div class="delete-button-column">
					<button type="button" class="delete-button text-danger btn-sm btn btn-link">
						<i class="bi bi-x-lg"></i>
					</button>
				</div>
			</div>
			<div class="forward-slash-warning" style="display:none;">
				<small class="text-danger">URI stubs must begin with a forward slash to be valid.</small>
			</div>
		`;

		const nameInput = template.querySelector(".virtual-parent-name-input");
		const uriInput = template.querySelector(".virtual-parent-uri-input");
		const virtualParentParentSelector = template.querySelector(".virtual-parent-parent-selector");
		const deleteButton = template.querySelector(".delete-button");

		// Change any <option> elements' textContent that is linked to this VirtualParent
		nameInput.addEventListener("input", () => {
			const nameValue = nameInput.value.trim();
			this.name = nameValue;
			for (const option of this.linkedOptions){
				option.textContent = nameValue;
			}
		});

		uriInput.addEventListener("input", () => {
			this.uri = uriInput.value.trim();

			if (!this.uri.startsWith("/")){
				this.showForwardSlashWarning();
			}else{
				this.hideForwardSlashWarning();
			}
		});

		// Preload in all the existing virtual parents
		for (const existingVirtualParentComponent of this.constructor.cache){
			if (existingVirtualParentComponent.id !== this.id) {
				const option = document.createElement("option");
				option.setAttribute("value", existingVirtualParentComponent.id);
				option.textContent = existingVirtualParentComponent.name;
				existingVirtualParentComponent.linkedOptions.push(option);
				virtualParentParentSelector.append(option);
			}
		}

		if (this.parentID !== null){
			virtualParentParentSelector.value = this.parentID;
		}

		virtualParentParentSelector.addEventListener("change", () => {
			this.parentID = parseInt(virtualParentParentSelector.value);
		});

		deleteButton.addEventListener("click", () => {
			this.constructor.removeByID(this.id);
		});

		this.constructor.container.append(template);

		return template;
	}

	showForwardSlashWarning(){
		const warningContainer = this.dom.querySelector(".forward-slash-warning");
		warningContainer.style.display = null;
	}

	hideForwardSlashWarning(){
		const warningContainer = this.dom.querySelector(".forward-slash-warning");
		warningContainer.style.display = "none";
	}
}

export default VirtualParent;