import ManagePageData from "../modals/ManagePageData.js";
import VirtualParent from "./VirtualParent.js";
import PageData from "./PageData.js";
import ContentFilter from "./ContentFilter.js";
import FeaturedImage from "../buttons/FeaturedImage.js";

class PageComponent{

	static nextID = 1;
	/** @type {PageComponent[]} */
	static cache = [];
	static pageContainers = {
		"General":document.querySelector("#general-pages-container"),
		"Service":document.querySelector("#service-pages-container"),
		"City":document.querySelector("#city-pages-container"),
		"Blog":document.querySelector("#blog-pages-container",),
		"Project":document.querySelector("#project-pages-container")
	};

	static getPagesOfType(pageType){
		const pages = [];
		for (const pageComponent of PageComponent.cache){
			if (pageComponent.pageType === pageType){
				pages.push(pageComponent);
			}
		}

		return pages;
	}

	id;
	pageName;
	pageType;
	slug;
	title;
	description;
	cityName;
	stateName;
	stateAbbreviation;
	featuredImageURI;
	featuredImageThumbURI;

	/** @type {null|int} */
	pageParentID = null;

	/** @type {PageData[]} */
	pageDatas = [];
	content;

	/**
	 * @param {{pageName: string, type: string, slug: string, seoTitle: string, seoDescription: string, content: string, featuredImageURI: string, featuredImageThumbURI: string, creationTimestamp: int, cityName: ?string, stateName: ?string, stateAbbreviation: ?string}} pageData
	 */
	constructor(pageData) {

		// Run the page content through a DOM manipulation filter
		const filteredContent = ContentFilter.filterContent(pageData.content);

		// Fix the slug if it doesn't start with a forward slash
		// if (!pageData.slug.startsWith("/")){
		// 	pageData.slug = `/${pageData.slug}`;
		// }

		this.id = this.constructor.nextID++;
		this.pageName = pageData.pageName;
		this.pageType = pageData.type;
		this.slug = pageData.slug;
		this.title = pageData.seoTitle;
		this.description = pageData.seoDescription;
		this.content = filteredContent;
		this.dom = this.getDOM();

		if (this.pageType === "City"){

			if ("cityName" in pageData){
				this.cityName = pageData.cityName;
				new PageData(this, "CITY_NAME", this.cityName);
			}

			if ("stateName" in pageData){
				this.stateName = pageData.stateName;
				new PageData(this, "STATE_NAME", this.stateName);
			}

			if ("stateAbbreviation" in pageData){
				this.stateAbbreviation = pageData.stateAbbreviation;
				new PageData(this, "STATE_NAME_SHORTHAND", this.stateAbbreviation);
			}
		}

		if(this.pageType !== "General") {
			this.featuredImageURI = pageData.featuredImageURI;
			this.featuredImageThumbURI = pageData.featuredImageThumbURI;
		}

		this.constructor.cache.push(this);
	}

	/**
	 * @returns {VirtualParent[]}
	 */
	getAncestorArray(){
		const ancestorVirtualParents = [];
		let currentParentID = this.pageParentID;
		console.log(`${this.pageName} Parent ID: ${currentParentID}`);
		while(currentParentID !== null){
			const virtualParent = VirtualParent.getByID(currentParentID);
			ancestorVirtualParents.push(virtualParent);
			if (virtualParent.parentID !== null) {
				currentParentID = virtualParent.parentID;
			}else{
				break;
			}
		}

		ancestorVirtualParents.reverse();

		return ancestorVirtualParents;
	}

	getDOM(event){
		const template = document.createElement("div");
		template.classList.add("page-component", "row");
		template.innerHTML = `
			<div class="col-xl-6">
				<div class="mb-1">
					<div class="input-group input-group-sm">
						<label for="page-name-input-${this.id}" class="input-group-text">
							<i class="bi bi-signpost-fill me-1"></i><span>Name</span>
						</label>
						<input id="page-name-input-${this.id}" type="text" class="form-control form-control-sm page-name-input" value="${this.pageName}">
					</div>
				</div>
				<div class="mb-1">
					<div class="input-group input-group-sm">
						<label for="page-route-input-${this.id}" class="input-group-text">
							<i class="bi bi-signpost-fill me-1"></i><span>Slug</span>
						</label>
						<input id="page-route-input-${this.id}" type="text" class="form-control form-control-sm page-route-input" value="${this.slug}">
						<select class="form-control page-parent-input">
							<option value="-1">- No Parent</option>
						</select>
					</div>
				</div>
				<div class="">
					<button type="button" class="manage-custom-data-button btn btn-light btn-sm rounded-0">
						<i class="bi bi-clipboard2-data-fill"></i>
						<span>Page Data</span>
					</button>
				</div>
			</div>
		`;

		// Insert the featured image if the page isn't General
		if (this.pageType !== "General"){
			template.innerHTML += `
				<div class="col-xl-6" id="page-featured-image">
					<input type="hidden" name="featured-image-uri" id="featured-image-uri-input-${this.id}" class="featured-image-uri-input" value="">
					<input type="hidden" name="featured-image-thumb-uri" id="featured-image-thumb-uri-input-${this.id}" class="featured-image-thumb-uri-input" value="">
					<div id="featured-image-contents-container">
						<div id="featured-image-container">
							<img id="featured-image-${this.id}" class="featured-image" alt="Featured image" src="">
						</div>
						<div id="featured-image-buttons">
							<div>
								<button type="button" class="mb-2 btn btn-primary set-featured-image-button">
									<i class="bi bi-image-fill"></i>
									<span>Set Image</span>
								</button>
							</div>
							<div>
								<button type="button" class="btn btn-danger clear-featured-image-button">
									<i class="bi bi-x-circle-fill"></i>
									<span>Clear</span>
								</button>
							</div>
						</div>
					</div>
				</div>
			`;
		}

		const pageNameInput = template.querySelector(".page-name-input");
		const pageRouteInput = template.querySelector(".page-route-input");
		const pageParentInput = template.querySelector(".page-parent-input");
		const manageCustomDataButton = template.querySelector(".manage-custom-data-button");

		const pageFeaturedImage = template.querySelector("#page-featured-image");
		const featuredImage = template.querySelector(".featured-image");
		const setFeaturedImageButton = template.querySelector(".set-featured-image-button");
		const clearFeaturedImageButton = template.querySelector(".clear-featured-image-button");

		const featuredImageURIInput = template.querySelector(".featured-image-uri-input");
		const featuredImageThumbURIInput = template.querySelector(".featured-image-thumb-uri-input");

		pageNameInput.addEventListener("input", () => {
			this.pageName = pageNameInput.value.trim();
		});

		pageRouteInput.addEventListener("input", () => {
			this.slug = pageRouteInput.value.trim();
		});

		pageParentInput.addEventListener("input", () => {
			const parentIDInt = parseInt(pageParentInput.value);

			if (isNaN(parentIDInt) || parentIDInt === -1){
				this.pageParentID = null;
			}else {
				this.pageParentID = parentIDInt;
			}
		});

		manageCustomDataButton.addEventListener("click", () => {
			ManagePageData.setCurrentPageComponent(this);
			ManagePageData.show();
		});

		if (pageFeaturedImage){
			featuredImage.addEventListener("click", () => {
				FeaturedImage.setImageButtonClicked(this.id);
			});

			setFeaturedImageButton.addEventListener("click", () => {
				FeaturedImage.setImageButtonClicked(this.id);
			});

			clearFeaturedImageButton.addEventListener("click", () => {
				FeaturedImage.clearFeaturedImage(this.id);
			});

			featuredImageURIInput.addEventListener("input", () => {
				this.featuredImageURI = featuredImageURIInput.value;
			});

			featuredImageThumbURIInput.addEventListener("input", () => {
				this.featuredImageThumbURI = featuredImageThumbURIInput.value;
			});
		}

		if (this.pageType === "General"){
			this.constructor.pageContainers.General.append(template);
		}else if (this.pageType === "Service"){
			this.constructor.pageContainers.Service.append(template);
		}else if (this.pageType === "City"){
			this.constructor.pageContainers.City.append(template);
		}else if (this.pageType === "Blog"){
			this.constructor.pageContainers.Blog.append(template);
		}else if (this.pageType === "Project"){
			this.constructor.pageContainers.Project.append(template);
		}

		return template;
	}
}

export default PageComponent;