import ConfirmModal from "../../utils/ConfirmModal.js";
import EditSitemapModal from "../modals/EditSitemapModal.js";
import EditSitemapForm from "../forms/EditSitemapForm.js";
import Endpoints from "../../Endpoints.js";
import SitemapManager from "../SitemapManager.js";

class Sitemap {

	static sitemapsContainer = document.querySelector("#sitemaps-rows-container");

	constructor(id, name, route, content, contentType){
		this.id = id;
		this.name = name;
		this.route = route;
		this.content = content;
		this.contentType = contentType;
		this.dom = this.getDOM();

		Sitemap.sitemapsContainer.append(this.dom);
	}

	getDOM(){
		const template = document.createElement("sitemap-row");
		template.classList.add("sitemaps-columns");

		template.innerHTML = `
			<div class="name">
				<span>${this.name}</span>
			</div>
			<div class="route">
				<span>${this.route}</span>
			</div>
			<div class="content-type">
				<span>${this.getContentTypeFromNumber(this.contentType)}</span>
			</div>
			<div class="management-buttons">
				<button type="button" class="edit-sitemap-button btn btn-sm btn-primary">
					<i class="bi bi-pencil-square"></i>
					<span>Edit</span>
				</button>
				<button type="button" class="delete-sitemap-button btn btn-sm btn-danger">
					<i class="bi bi-trash-fill"></i>
					<span>Delete</span>
				</button>
			</div>
		`;

		const editButton = template.querySelector(".edit-sitemap-button");
		editButton.addEventListener("click", () => {
			this.editSitemap();
		});

		const deleteButton = template.querySelector(".delete-sitemap-button");
		deleteButton.addEventListener("click", () => {
			this.deleteSitemap();
		});

		return template;
	}

	/**
	 * Gets the content type from the number
	 */
	getContentTypeFromNumber(number){
		const contentTypes = [
			[0, "XML"],
			[1, "RSS"],
			[2, "Text"],
		];

		for(const type of contentTypes) {
			if(number === type[0]) {
				return type[1];
			}
		}
	}

	/**
	 * Sets up the edit modal and shows it
	 */
	editSitemap(){
		EditSitemapModal.currentSitemapID = this.id;

		// Setup the fields
		SitemapManager.editSitemapAceEditor.session.setValue(this.content);
		EditSitemapForm.form.querySelector(`[name="sitemap-name"]`).value = this.name;
		EditSitemapForm.form.querySelector(`[name="sitemap-route"]`).value = this.route;
		EditSitemapForm.form.querySelector(`[name="sitemap-content-type"]`).value = this.contentType;
		EditSitemapModal.showModal();
	}

	/**
	 * Submits a request to delete the sitemap
	 */
	async deleteSitemap(){
		const deleteModal = new ConfirmModal("delete-sitemap-modal");
		deleteModal.setConfirmButtonText("Delete");
		deleteModal.setTitle("Delete this sitemap?");
		deleteModal.setContent("Please confirm you'd like to permanently delete this sitemap.");
		deleteModal.showModal();
		const didDelete = await deleteModal.actionTaken();
		deleteModal.cleanup();

		if (didDelete){
			const response = await fetch(`${Endpoints.sitemapManager.deleteSitemap}/${this.id}`, {
				method:"DELETE",
				cache:"no-cache",
				credentials:"same-origin"
			});

			/** @type {{error: ?string, status: int}} */
			let data;
			try{
				data = await response.json();
			}catch(jsonSyntaxError){
				alert("The server responded with invalid JSON.");
			}

			if (data.status === 1){
				this.dom.remove();
			}else if (data.status === -1){
				alert(data.error);
			}
		}
	}
}

export default Sitemap;