import ReadFullReviewModal from "../modals/ReadFullReviewModal.js";
import ConfirmModal from "../../utils/ConfirmModal.js";
import Endpoints from "../../Endpoints.js";
import EditReviewModal from "../modals/EditReviewModal.js";
import EditReviewForm from "../forms/EditReviewForm.js";

class Review{

	static reviewsContainer = document.querySelector("#review-rows-container");

	constructor(id, firstName, lastName, city, state, rating, body, timestamp){
		this.id = id;
		this.firstName = firstName;
		this.lastName = lastName;
		this.city = city;
		this.state = state;
		this.rating = rating;
		this.body = body;
		this.timestamp = timestamp;
		this.dom = this.getDOM();

		Review.reviewsContainer.append(this.dom);
	}

	getDOM(){
		const template = document.createElement("review-component");
		template.classList.add("review-columns");
		const date = (new Date(parseInt(this.timestamp) * 1000)).toLocaleDateString();
		let locationString = "";

		if (this.city && this.city !== ""){
			locationString += this.city;
		}

		// If there is no city set, but there is a state
		if (locationString.length === 0 && this.state && this.state !== "") {
			locationString = this.state;
		}else{
			// If there is a city set and there is a state
			if (locationString.length > 0 && this.state && this.state !== ""){
				locationString += `, ${this.state}`;
			}
		}

		// No city or state
		if (locationString.length === 0){
			locationString = "[No location]";
		}

		// Rating stars
		const starHTML = `<i class="bi bi-star-fill"></i>`;
		let ratingsHTML = "[No rating]";

		if (this.rating && this.rating > 0){
			ratingsHTML = starHTML.repeat(this.rating);
		}

		// Name handling
		let customerName = this.firstName;
		if (this.lastName && this.lastName.length > 0){
			customerName += ` ${this.lastName}`;
		}

		// Handle body truncation
		let reviewWasTruncated = false;
		let reviewBody = this.body;
		if (reviewBody.length > 200){
			reviewBody = `${this.body.substring(0, 199)}[...]`;
			reviewWasTruncated = true;
		}

		template.innerHTML = `
			<div class="timestamp">
				<span>${date}</span>
			</div>
			<div class="location">
				<span>${locationString}</span>
			</div>
			<div class="rating">
				<strong>${this.rating}/5</strong> <span class="stars">${ratingsHTML}</span>
			</div>
			<div class="name">
				<span>${customerName}</span>
			</div>
			<div class="body">
				<span>${reviewBody}</span>
			</div>
			<div class="management-buttons">
				<button type="button" class="edit-review-button btn btn-sm btn-primary">
					<i class="bi bi-pencil-square"></i>
					<span>Edit</span>
				</button>
				<button type="button" class="delete-review-button btn btn-sm btn-danger">
					<i class="bi bi-trash-fill"></i>
					<span>Delete</span>
				</button>
			</div>
		`;

		// Add a read more button to read the full review if it was truncated
		if (reviewWasTruncated){
			const bodyContainer = template.querySelector(".body");
			const breakElement = document.createElement("br");
			const readMoreButton = document.createElement("button");
			readMoreButton.setAttribute("type", "button");
			readMoreButton.classList.add("btn");
			readMoreButton.classList.add("btn-sm");
			readMoreButton.classList.add("p-0");
			readMoreButton.classList.add("btn-link");
			readMoreButton.textContent = "Full review";
			bodyContainer.append(breakElement);
			bodyContainer.append(readMoreButton);

			readMoreButton.addEventListener("click", () => {
				ReadFullReviewModal.testimonialBody.textContent = this.body;
				ReadFullReviewModal.showModal();
			});
		}

		const editButton = template.querySelector(".edit-review-button");
		editButton.addEventListener("click", () => {
			this.editReview();
		});

		const deleteButton = template.querySelector(".delete-review-button");
		deleteButton.addEventListener("click", () => {
			this.deleteReview();
		});

		return template;
	}

	/**
	 * Sets up the edit modal and shows it
	 */
	editReview(){
		EditReviewModal.currentReviewID = this.id;

		// Setup the fields
		EditReviewModal.ratingStars.parentElement.setAttribute("rating", this.rating);
		EditReviewModal.ratingInput.value = this.rating;
		EditReviewModal.ratingStars.highlightToRating(this.rating);
		EditReviewForm.form.querySelector(`[name="first-name"]`).value = this.firstName;
		EditReviewForm.form.querySelector(`[name="last-name"]`).value = this.lastName;
		EditReviewForm.form.querySelector(`[name="city"]`).value = this.city;
		EditReviewForm.form.querySelector(`[name="state"]`).value = this.state;
		EditReviewForm.form.querySelector(`[name="body"]`).value = this.body;
		EditReviewModal.showModal();
	}

	/**
	 * Submits a request to delete the review
	 */
	async deleteReview(){
		const deleteModal = new ConfirmModal("delete-review-modal");
		deleteModal.setConfirmButtonText("Delete");
		deleteModal.setTitle("Delete this review?");
		deleteModal.setContent("Please confirm you'd like to permanently delete this review.");
		deleteModal.showModal();
		const didDelete = await deleteModal.actionTaken();
		deleteModal.cleanup();

		if (didDelete){
			const response = await fetch(`${Endpoints.reviews.deleteReview}/${this.id}`, {
			    method:"DELETE",
			    cache:"no-cache",
			    credentials:"same-origin"
			});

			/** @type {{error: ?string, status: int}} */
			let data;
			try{
			    data = await response.json();
			}catch(jsonSyntaxError){
			    alert("The server responded with invalid JSON.");
			}

			if (data.status === 1){
				this.dom.remove();
			}else if (data.status === -1){
				alert(data.error);
			}
		}
	}
}

export default Review;