import Review from "./components/Review.js";
import Endpoints from "../Endpoints.js";
import NewReviewModal from "./modals/NewReviewModal.js";
import NewReviewForm from "./forms/NewReviewForm.js";
import SearchReviewsForm from "./forms/SearchReviewsForm.js";
import Paginator from "../utils/Paginator.js";
import Scheduler from "../utils/Scheduler.js";

class ManageReviews{

	static DEFAULT_PAGE = 1;
	static DEFAULT_LIMIT = 10;

	currentPage = 1;
	currentLimit = 10;
	currentQuery = "";
	loaderElement = document.querySelector("#reviews-loader");
	noResultsMessageContainer = document.querySelector("#no-reviews-message-container");

	constructor(){
		/**
		 * @type {Paginator[]}
		 */
		this.paginators = [];

		for (const paginator of document.querySelectorAll("pagination")){
			const newPaginator = new Paginator(paginator);
			this.hookPaginatorEvents(newPaginator);
			this.paginators.push(newPaginator);
		}

		(async () => {
			this.loadReviews(this.currentPage, this.currentLimit, null);
		})();
	}

	/**
	 *
	 * @param {Paginator} paginator
	 */
	hookPaginatorEvents(paginator){
		paginator.onPagePrev(pageNumber => {
			this.currentPage = pageNumber;
			this.loadReviews(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});

		paginator.onPageNext(pageNumber => {
			this.currentPage = pageNumber;
			this.loadReviews(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});

		paginator.onPageManuallyEntered(pageNumber => {
			this.currentPage = pageNumber;
			this.loadReviews(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});
	}

	/**
	 * @param {int} page
	 * @param {int} limit
	 * @param {string|null} query
	 * @returns {Promise<void>}
	 */
	async loadReviews(page, limit, query){
		const urlParams = new URLSearchParams();
		urlParams.set("page", String(page));
		urlParams.set("limit", String(limit));

		if (query !== null) {
			urlParams.set("query", query);
		}

		// Clear current reviews
		Review.reviewsContainer.innerHTML = "";
		this.noResultsMessageContainer.style.display = "none";

		// Show loader
		this.loaderElement.style.display = "block";

		const response = await fetch(`${Endpoints.reviews.getReviews}?${urlParams.toString()}`, {
		    body:null,
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		let data;
		try{
			/** @type {{status: int, error: ?string, reviews: Object[], totalReviews: int, totalPages: int}} */
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		}

		// Force-wait a minimum of 0.35s so things don't flash
		await Scheduler.wait(350);

		// Hide loader
		this.loaderElement.style.display = "none";

		if (data.status === 1){

			// Set paginator stuff
			for (const paginator of this.paginators){
				// paginator.setCurrentPage(this.currentPage);
				paginator.setMaxPages(data.totalPages);
			}

			if (data.reviews.length > 0) {
				for (/** @type {{id: int, firstName: string, lastName: string, rating: int, postedTimestamp: int, body: string, city: string, state: string}} */ const review of data.reviews) {
					new Review(
						review.id,
						review.firstName,
						review.lastName,
						review.city,
						review.state,
						review.rating,
						review.body,
						review.postedTimestamp
					);
				}
			}else{
				this.noResultsMessageContainer.style.display = null;
			}
		}else if (data.status === -1){

		}
	}
}

export default new ManageReviews();