import NewRedirectModal from "./modals/NewRedirectModal.js";
import NewRedirectForm from "./forms/NewRedirectForm.js";
import EditRedirectForm from "./forms/EditRedirectForm.js";
import SearchRedirectsForm from "./forms/SearchRedirectsForm.js";
import Endpoints from "../Endpoints.js";
import Redirect from "./components/Redirect.js";
import Scheduler from "../utils/Scheduler.js";
import Paginator from "../utils/Paginator.js";

class ManageRedirects{

	currentPage = 1;
	currentLimit = 25;
	currentQuery = "";
	loader = document.querySelector("#redirects-loader");
	noResultsMessageContainer = document.querySelector("#no-redirects-message-container");
	isProcessing = false;

	/**
	 * @type {Paginator[]}
	 */
	paginators = [];

	constructor(){

		for (const paginator of document.querySelectorAll("pagination")){
			const newPaginator = new Paginator(paginator);
			this.hookPaginatorEvents(newPaginator);
			this.paginators.push(newPaginator);
		}

		this.loadRedirects(
			this.currentPage,
			this.currentLimit,
			this.currentQuery
		);
	}

	/**
	 *
	 * @param {Paginator} paginator
	 */
	hookPaginatorEvents(paginator){
		paginator.onPagePrev(pageNumber => {
			this.currentPage = pageNumber;
			this.loadRedirects(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});

		paginator.onPageNext(pageNumber => {
			this.currentPage = pageNumber;
			this.loadRedirects(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});

		paginator.onPageManuallyEntered(pageNumber => {
			this.currentPage = pageNumber;
			this.loadRedirects(
				pageNumber,
				this.currentLimit,
				this.currentQuery
			)
		});
	}

	/**
	 * @param {int} page
	 * @param {int} limit
	 * @param {string} query
	 * @returns {Promise<void>}
	 */
	async loadRedirects(page, limit, query){

		if (this.isProcessing){
			return;
		}

		this.noResultsMessageContainer.style.display = "none";
		Redirect.redirectsContainer.innerHTML = "";

		this.isProcessing = true;
		this.loader.style.display = "block";
		this.currentPage = page;
		this.currentLimit = limit;
		this.currentQuery = query;

		const urlParams = new URLSearchParams();
		urlParams.set("limit", String(limit));
		urlParams.set("page", String(page));
		urlParams.set("query", query);

		const response = await fetch(`${Endpoints.redirects.getRedirects}?${urlParams.toString()}`, {
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		await Scheduler.wait(350);

		let data;
		try{
		    /** @type {{status: int, error: ?string, redirects: Object[], totalRedirects: int, totalPages: int}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		    return;
		}

		this.loader.style.display = "none";

		if (data.status === 1){

			// Set paginator stuff
			for (const paginator of this.paginators){
				// paginator.setCurrentPage(this.currentPage);
				paginator.setMaxPages(data.totalPages);
			}

			if (data.redirects.length > 0) {
				for (/** @type {{statusCode: int, id: int, isRegEx: int, fromURI: string, to: string, addedTimestamp: int, preserveQueryString: int}} */ const redirect of data.redirects) {
					new Redirect(
						redirect.id,
						redirect.isRegEx,
						redirect.fromURI,
						redirect.to,
						redirect.statusCode,
						redirect.addedTimestamp,
						redirect.preserveQueryString
					);
				}
			}else{
				this.noResultsMessageContainer.style.display = null;
			}
		}else if (data.status === -1){
			alert(data.error);
		}

		this.isProcessing = false;
	}
}

export default new ManageRedirects();