import Endpoints from "../../Endpoints.js";
import WindowManager from "../WindowManager.js";
import DirectoryImagesFetcher from "../DirectoryImagesFetcher.js";
import SelectionListener from "../SelectionListener.js";
import ImageManagerState from "../ImageManagerState.js";
import MetaDataPane from "../MetaDataPane.js";
import MainWindow from "./MainWindow.js";

class ResizeWindow{

	static previewMarginShift = "90"; // In percent

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @type {HTMLElement}
	 */
	window;

	constructor(imageManager){
		this.imageManager = imageManager;
		this.window = this.imageManager.container.querySelector(".im-resize-window");
		this.dismissButton = this.window.querySelector(".im-modal-dismiss-button");
		this.form = this.window.querySelector(".im-resize-fields");
		this.formSubmitButton = this.window.querySelector(".im-resize-form-submit-button");
		this.previewsContainer = this.window.querySelector(".im-resize-preview-images-container");
		this.spinnerContainer = this.window.querySelector(".resize-modal-loading-spinner");
		this.constrainCheckbox = this.window.querySelector(`.im-resize-constrained-checkbox`);
		this.imageFilePathInput = this.window.querySelector(`[name="image-file-path"]`);
		this.oWidthInput = this.window.querySelector(`[name="original-width"]`);
		this.widthInput = this.window.querySelector(`[name="width"]`);
		this.oHeightInput = this.window.querySelector(`[name="original-height"]`);
		this.heightInput = this.window.querySelector(`[name="height"]`);
		this.leftPreviewShiftbutton = this.window.querySelector(".im-resize-scroll-left-button");
		this.rightPreviewShiftbutton = this.window.querySelector(".im-resize-scroll-right-button");

		/** @type {ImageComponent[]} Because not all images in the current selection are valid resizables */
		this.queuedImageComponents = [];

		/** @type {int} */
		this.currentPreviewIndex = 0;

		/** @type {boolean} Flag for when the form is submitting and being processed */
		this.isProcessing = false;

		/** @type {HTMLElement[]} */
		this.currentPreviews = [];

		this.heightInput.addEventListener("input", () => {
			this.onSizeInputChanged(this.heightInput);
		});

		this.widthInput.addEventListener("input", () => {
			this.onSizeInputChanged(this.widthInput);
		});

		this.formSubmitButton.addEventListener("click", () => {
			this.onSubmit();
		});

		this.form.addEventListener("submit",e => {
			e.preventDefault();
			this.onSubmit();
		});

		this.dismissButton.addEventListener("click", () => {
			if (!this.isProcessing){
				this.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);
			}
		});

		this.leftPreviewShiftbutton.addEventListener("click", e => {
			this.shiftPreview("left");
		});

		this.rightPreviewShiftbutton.addEventListener("click", e => {
			this.shiftPreview("right");
		});
	}

	/**
	 * Gather valid resizable images
	 */
	getAcceptableImagesFromSelection(){
		this.queuedImageComponents = [];
		for (/** @type {ImageComponent} */ const component of this.imageManager.selectionListener.currentImageComponentSelection){
			if (component.fileExtension !== "svg"){
				this.queuedImageComponents.push(component);
			}
		}
	}

	/**
	 * Shifts the preview a direction
	 */
	shiftPreview(direction){
		if (direction === "left") {
			if (this.currentPreviewIndex > 0) {
				--this.currentPreviewIndex;
			}
		}else if(direction === "right"){
			if (this.currentPreviewIndex < this.queuedImageComponents.length - 1){
				++this.currentPreviewIndex;
			}
		}

		this.window.querySelector(".im-resize-current-index").textContent = String(this.currentPreviewIndex + 1);
		this.previewsContainer.style.marginLeft = `-${this.currentPreviewIndex * ResizeWindow.previewMarginShift}%`;
	}

	/**
	 * When the resize form is submitted
	 * @return {string[]} Of errors
	 */
	async onSubmit(){
		if (this.isProcessing){
			return;
		}

		const currentQueueNumber = this.spinnerContainer.querySelector(".im-finished-resize-number");

		this.isProcessing = true;
		this.spinnerContainer.querySelector(".im-resize-in-queue-upload-number").textContent = String(this.queuedImageComponents.length);
		currentQueueNumber.textContent = "0";
		this.spinnerContainer.style.display = "flex";

		let finishedNumber = 0;
		const errors = [];
		for (/** @type {ImageComponent} */ const imageComponent of this.queuedImageComponents){
			await new Promise(async resolve => {

				const fData = new FormData();
				const aspectRatio = parseInt(imageComponent.imageWidth) / parseInt(imageComponent.imageHeight);
				let widthForImage = parseInt(this.widthInput.value);
				let heightForImage = parseInt(this.heightInput.value);

				if (isNaN(heightForImage) && isNaN(widthForImage)){
					errors.push(`${imageComponent.fileName}: Both input values cannot be invalid numbers or blank.`);
					return resolve();
				}else if (isNaN(widthForImage) && !isNaN(heightForImage)){
					// Use the height to get the width
					widthForImage = heightForImage / aspectRatio;
				}else if (isNaN(heightForImage) && !isNaN(widthForImage)){
					// Use the width to get the height for the image
					heightForImage = aspectRatio * widthForImage;
				}

				fData.set("width", String(widthForImage));
				fData.set("height", String(heightForImage));
				fData.set("image-file-path", imageComponent.fullFilePath);

				const response = await fetch(`/uplift/image-manager/resize-image`, {
					credentials:"same-origin",
					cache:"no-cache",
					body:fData,
					method:"PATCH"
				});

				if (response.status === 200) {
					/** @type {{status: int, error:?string}} **/
					let data;
					try {
						data = await response.json();
					}catch(jsonSyntaxError){
						errors.push("The server responded with invalid JSON.");
					}
					if (data.status === -1) {
						errors.push(data.error);
					}
				}else{
					errors.push(response.statusText);
				}

				resolve();
			});
			++finishedNumber;
			currentQueueNumber.textContent = String(finishedNumber);
		}

		// Refresh the cache string for previews
		this.imageManager.imageManagerState.currentImageCacheRandomString = String(Math.random());

		await this.imageManager.windowManager.mainWindow.refreshCurrentLocation();
		this.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);

		this.isProcessing = false;
		this.spinnerContainer.style.display = "none";

		if (errors.length > 0){
			alert("There were errors with some or all of the images processed. Check the developer console for details.");
			console.log(errors);
		}

		return errors;
	}

	/**
	 * Function called when this window gets shown
	 */
	onWindowShown(){
		this.getAcceptableImagesFromSelection();
		this.window.querySelector(".im-resize-max-index").textContent = String(this.queuedImageComponents.length);

		// Reset old stored values from a previous operation
		this.previewsContainer.innerHTML = "";
		this.window.querySelector(".im-resize-current-index").textContent = "1";
		this.currentPreviewIndex = 0;
		this.previewsContainer.style.marginLeft = `-${this.currentPreviewIndex * ResizeWindow.previewMarginShift}%`;

		this.widthInput.value = "";
		this.heightInput.value = "";

		if (this.queuedImageComponents.length === 1){
			/** @type {ImageComponent} */
			const imageComponentToBeResized = this.queuedImageComponents[0];
			this.constrainCheckbox.checked = true;
			this.constrainCheckbox.parentElement.style.display = "inline-flex";
			this.oHeightInput.value = imageComponentToBeResized.imageHeight;
			this.oWidthInput.value = imageComponentToBeResized.imageWidth;
			this.heightInput.value = imageComponentToBeResized.imageHeight;
			this.widthInput.value = imageComponentToBeResized.imageWidth;
		}else if (this.queuedImageComponents.length > 1){
			this.constrainCheckbox.checked = false;
			this.constrainCheckbox.parentElement.style.display = "none";
		}else{
			// Shouldn't happen
			console.log("Valid image component selection is empty!");
		}

		this.buildImagePreviews();
	}

	/**
	 * @param {string} imageSource
	 * @return {HTMLElement}
	 */
	buildImagePreviewDOM(imageSource){
		const template = document.createElement("im-resize-preview")
		template.innerHTML = `
			<img class="im-resize-preview-image" alt="Preview of image after resizing" src="${imageSource}?cache=${this.imageManager.imageManagerState.currentImageCacheRandomString}">
		`;

		this.currentPreviews.push(template);
		this.previewsContainer.append(template);

		return template;
	}

	/**
	 * Builds the image previews based on the currently selected image components
	 */
	buildImagePreviews(){
		this.currentPreviews = [];
		for (/** @type {ImageComponent} */ const component of this.queuedImageComponents){
			this.buildImagePreviewDOM(component.uri);
		}
	}

	/**
	 * Refreshes all of the image previews with the new sizes
	 * @param {HTMLInputElement} inputElementChanged Will have a name of "width" or "height"
	 */
	refreshPreviewsWithNewSizes(inputElementChanged){
		const nameOfInputElementChanged = inputElementChanged.getAttribute("name");
		const heightInput = parseInt(this.heightInput.value);
		const widthInput = parseInt(this.widthInput.value);

		if (this.queuedImageComponents.length > 1) {
			for (/** @type {HTMLElement} */ const previewContainer of this.currentPreviews) {
				const preview = previewContainer.querySelector("img");
				if (!isNaN(heightInput) && !isNaN(widthInput)) {
					preview.setAttribute("width", `${widthInput}px`);
					preview.setAttribute("height", `${heightInput}px`);
				} else {
					if (!isNaN(heightInput)) {
						preview.removeAttribute("width");
						preview.setAttribute("height", `${heightInput}px`);
					} else if (!isNaN(widthInput)) {
						preview.removeAttribute("height");
						preview.setAttribute("width", `${widthInput}px`);
					} else {
						// Neither?
						preview.removeAttribute("height");
						preview.removeAttribute("width");
					}
				}
			}
		}else if (this.queuedImageComponents.length === 1){
			/** @type {HTMLImageElement} */
			const preview = this.currentPreviews[0].querySelector("img");

			// Only a single image change. Check for aspect ratio retention
			if (this.constrainCheckbox.checked) {
				const aspectRatio = parseInt(this.oHeightInput.value) / parseInt(this.oWidthInput.value);
				if (nameOfInputElementChanged === "height"){
					this.widthInput.value = Math.round(inputElementChanged.value / aspectRatio);
				}else if (nameOfInputElementChanged === "width"){
					this.heightInput.value = Math.round(aspectRatio * inputElementChanged.value);
				}

				// Only set the height attribute to avoid stretching the preview image when aspect ratio is constrained
				preview.removeAttribute("width");
				preview.setAttribute("height", `${this.heightInput.value}px`);
			}else{
				// No aspect ratio retention
				// Set both height and width when no constraint is applied
				preview.setAttribute("width", `${this.widthInput.value}px`);
				preview.setAttribute("height", `${this.heightInput.value}px`);
			}
		}
	}

	/**
	 * When any size input boxes are changed
	 * @param {HTMLInputElement} inputElementChanged
	 */
	onSizeInputChanged(inputElementChanged){
		this.refreshPreviewsWithNewSizes(inputElementChanged);
	}
}

export default ResizeWindow;