import WindowManager from "../WindowManager.js";
import Endpoints from "../../Endpoints.js";
import ImageManagerState from "../ImageManagerState.js";
import DirectoryImagesFetcher from "../DirectoryImagesFetcher.js";
import SelectionListener from "../SelectionListener.js";

class MainWindow{
	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @type {HTMLElement}
	 */
	imagePreviewLoader;

	/**
	 * @type {HTMLElement}
	 */
	imagesLoadingContainer;

	/**
	 * @type {HTMLElement}
	 */
	noImagesMessageContainer;

	/**
	 * @param {ImageManager} imageManager
	 */
	constructor(imageManager){
		this.imageManager = imageManager;
		this.imagePreviewLoader = this.imageManager.container.querySelector(".image-preview-loader");
		this.imagesLoadingContainer = this.imageManager.container.querySelector(".im-images-loading-container");
		this.noImagesMessageContainer = this.imageManager.container.querySelector(".im-no-images-message");
		this.window = document.querySelector("image-manager-window");
		this.windowBody = this.window.querySelector("im-window-body");
		this.windowFolderContents = this.windowBody.querySelector(".folder-contents-container");
		this.dragOverUploadWindow = this.window.querySelector("#im-upload-container");
		this.dragOverEventDepth = 0;
		this.dismissButton = this.window.querySelector(".im-modal-dismiss-button");
		this.isProcessing = false;

		this.dismissButton.addEventListener("click", () => {
			if (!this.isProcessing){
				this.imageManager.windowManager.closeImageManager();
			}
		});

		this.windowFolderContents.addEventListener("dragover",
			/** @param {DragEvent} e */
			(e) => {
				e.preventDefault();
			}
		);

		this.windowFolderContents.addEventListener("dragenter",
			/** @param {DragEvent} e */
			(e) => {
				e.preventDefault();
				if (!this.imageManager.imageManagerState.isDraggingComponent) {
					++this.dragOverEventDepth;
					this.onWindowBodyFolderContentsDragEnter(e);
				}
			}
		);

		this.windowFolderContents.addEventListener("dragleave",
			/** @param {DragEvent} e */
			(e) => {
				e.preventDefault();
				if (!this.imageManager.imageManagerState.isDraggingComponent) {
					--this.dragOverEventDepth;
					if (this.dragOverEventDepth === 0) {
						this.onWindowBodyFolderContentsDragLeave(e);
					}
				}
			}
		);

		this.dragOverUploadWindow.addEventListener("drop",
			/** @param {DragEvent} e */
			(e) => {
				e.preventDefault();
				// Reset the event depth
				this.dragOverEventDepth = 0;
				this.onUploadWindowDrop(e);
			}
		);
	}

	/**
	 * When the image manager's main window body folder contents receives a drag enter event
	 * @param {DragEvent} e
	 */
	onWindowBodyFolderContentsDragEnter(e){
		this.showUploadWindow();
	}

	/**
	 * When the image manager's main window body folder contents has had the drag operation leave the target zone
	 * @param {DragEvent} e
	 */
	onWindowBodyFolderContentsDragLeave(e){
		this.hideUploadWindow();
	}

	/**
	 * When a drop event fires on the upload window
	 * @param {DragEvent} e
	 */
	async onUploadWindowDrop(e){
		this.isProcessing = true;

		// Check if the user is dragging an image from somewhere else
		const htmlUpload = e.dataTransfer.getData("text/html");
		if (htmlUpload !== null){
			if (htmlUpload.trim() !== ""){
				const domParser = new DOMParser();
				const htmlDoc = domParser.parseFromString(htmlUpload, 'text/html');
				const img = htmlDoc.querySelector("img");
				if (img !== null){
					const source = img.getAttribute("src");
					if (source !== null){
						if (source.trim() !== ""){
							// It is an HTML image upload

							// Show the uploading components
							this.showUploadingInProgressElements();

							await this.uploadImageBySource(source);

							// Make sure to return here to not process other if conditions
							// Reload the images location
							this.imageManager.directoryImagesFetcher.setDirectory(this.imageManager.imageManagerState.currentDirectory);
							await this.imageManager.directoryImagesFetcher.fetchAndRender();
							this.imageManager.metaDataPane.hidePreview();
							this.imageManager.metaDataPane.hideImageData();
							this.hideUploadWindow();
							return;
						}
					}
				}
			}
		}

		// Probably not an HTML image upload if gotten here.
		// Check for file transfers
		const files = e.dataTransfer.files;
		if (files.length > 0){


			const errors = await this.uploadListOfFiles(files);

			// Set processing to false. refreshCurrentLocation will set it back to true
			this.isProcessing = false;

			// All uploads are finished
			await this.refreshCurrentLocation();
			this.hideUploadWindow();

			if (errors.length > 0) {
				alert("There were upload errors. Check the developer console.");
				console.log("--- Upload errors ---");
				console.log(errors);
			}
		}

		this.isProcessing = false;
	}

	/**
	 * Refreshes the current location
	 */
	async refreshCurrentLocation(){
		if (this.isProcessing){
			return;
		}

		// Clear the selection
		this.imageManager.selectionListener.clearImageComponentSelection();
		this.isProcessing = true;

		this.imageManager.directoryImagesFetcher.setDirectory(this.imageManager.imageManagerState.currentDirectory);
		await this.imageManager.directoryImagesFetcher.fetchAndRender();
		this.imageManager.metaDataPane.hidePreview();
		this.imageManager.metaDataPane.hideImageData();

		this.isProcessing = false;
	}

	/**
	 * Uploads a list of files
	 * @param {FileList} files
	 * @return {Promise<array>}
	 */
	async uploadListOfFiles(files){
		const finishedUploadSpan = this.dragOverUploadWindow.querySelector("#im-finished-upload-number");
		this.dragOverUploadWindow.querySelector("#im-in-queue-upload-number").textContent = String(files.length);
		finishedUploadSpan.textContent = "0";
		let filesUploaded = 0;
		let errors = [];
		this.showUploadingInProgressElements();

		for (const file of files) {
			const thisPromise = new Promise(async resolve => {
				// const fileReadPromise = new Promise((fileReadResolve, fileReadReject) => {
				// 	const reader = new FileReader();
				// 	reader.onload = () => {
				// 		fileReadResolve(reader.result);
				// 	};
				//
				// 	reader.onerror = () => {
				// 		fileReadReject(reader.error);
				// 	};
				//
				// 	reader.readAsDataURL(file);
				// });
				//
				// const fileAsBinary = await fileReadPromise;
				try {
					await this.uploadBinaryImage(file.name, file);
				} catch (uploadError) {
					errors.push(uploadError);
				}

				resolve();
			});

			await thisPromise;
			++filesUploaded;
			finishedUploadSpan.textContent = String(filesUploaded);
		}

		return errors;
	}

	/**
	 * Uploads an image by its remote URL
	 * @param {string} source Remote URL
	 */
	async uploadImageBySource(source){
		const fData = new FormData();
		fData.set("to-directory", this.imageManager.imageManagerState.currentDirectory);
		fData.set("image-source", source);

		const response = await fetch(`/uplift/image-manager/upload-image-by-url-source`, {
			credentials:"same-origin",
			body:fData,
			cache:"no-cache",
			method:"post"
		});

		if (response.status === 200){
			const data = await response.json();
			if (data.status === 1){

			}else{
				throw data.error;
			}
		}else{
			throw "Unhandled server error.";
		}
	}

	/**
	 * Uploads a binary string to the upload image file endpoint
	 * @param {string} fileName
	 * @param {Blob} file
	 */
	async uploadBinaryImage(fileName, file){
		const fData = new FormData();
		fData.set("to-directory", this.imageManager.imageManagerState.currentDirectory);
		fData.set("file-binary", file, fileName);

		const response = await fetch(`/uplift/image-manager/upload-image-file`, {
			credentials:"same-origin",
			body:fData,
			cache:"no-cache",
			method:"post"
		});

		if (response.status === 200){
			const data = await response.json();
			if (data.status === 1){

			}else{
				throw data.error;
			}
		}else{
			throw "Unhandled server error.";
		}
	}

	/**
	 * Shows the necessary elements in the upload window to display to the user that an upload is in progress
	 */
	showUploadingInProgressElements(){
		this.dragOverUploadWindow.querySelector("#im-finished-upload-number").textContent = "0";
		this.dragOverUploadWindow.querySelector("#im-uploading-spinner").style.display = "block";
		this.dragOverUploadWindow.querySelector(".im-upload-icon-container").style.display = "none";
		this.dragOverUploadWindow.querySelector("#im-upload-before-drop").style.display = "none";
		this.dragOverUploadWindow.querySelector("#im-upload-after-drop").style.display = "block";
	}

	/**
	 * Shows the upload window
	 */
	showUploadWindow(){
		this.dragOverUploadWindow.querySelector("#im-uploading-spinner").style.display = "none";
		this.dragOverUploadWindow.querySelector(".im-upload-icon-container").style.display = "block";
		this.dragOverUploadWindow.querySelector("#im-upload-before-drop").style.display = "block";
		this.dragOverUploadWindow.querySelector("#im-upload-after-drop").style.display = "none";
		this.dragOverUploadWindow.style.display = "flex";
	}
	/**
	 * Hides the upload window
	 */
	hideUploadWindow(){
		this.dragOverUploadWindow.style.display = "none";
	}

	/**
	 * Shows the spinning loader in the image data right-hand panel.
	 */
	showImagePreviewLoader(){
		this.imagePreviewLoader.style.display = null;
	}

	/**
	 * Hide the spinning loader in the image data right-hand panel.
	 */
	hideImagePreviewLoader(){
		this.imagePreviewLoader.style.display = "none";
	}

	/**
	 * Shows the overlay that display a spinner and text representing images are populating/loading in.
	 */
	showImagesLoading(){
		this.imagesLoadingContainer.style.display = null;
	}

	/**
	 * Hides the overlay that display a spinner and text representing images are populating/loading in.
	 */
	hideImagesLoading(){
		this.imagesLoadingContainer.style.display = "none";
	}

	/**
	 * Shows a message displaying no images in this directory.
	 */
	showNoImagesMessage(){
		this.noImagesMessageContainer.style.display = null;
	}

	/**
	 * Hides a message displaying no images in this directory.
	 */
	hideNoImagesMessage(){
		this.noImagesMessageContainer.style.display = "none";
	}

}

export default MainWindow;