import WindowManager from "../WindowManager.js";
import DirectoryImagesFetcher from "../DirectoryImagesFetcher.js";
import ImageManagerState from "../ImageManagerState.js";
import Scheduler from "../../utils/Scheduler.js";

class CropWindow{

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	constructor(imageManager){
		this.imageManager = imageManager;
		this.window = this.imageManager.container.querySelector(".im-crop-window");
		this.guidelinesBackground = this.window.querySelector(".crop-guidelines-background");
		this.guidelinesContainer = this.window.querySelector(".crop-guidelines-container");
		this.dismissButton = this.window.querySelector(".im-modal-dismiss-button");
		this.form = this.window.querySelector(".im-crop-fields");
		this.formSubmitButton = this.window.querySelector(".im-crop-fields-submit-button");
		this.previewContainer = this.window.querySelector(".im-crop-preview-container");
		this.previewImage = this.window.querySelector(".im-crop-preview-image");
		this.spinner = this.window.querySelector(".im-crop-loading-spinner");
		this.imageFilePathInput = this.window.querySelector(`[name="image-file-path"]`);
		this.ratioXInput = this.window.querySelector(`[name="ratio-x"]`);
		this.ratioYInput = this.window.querySelector(`[name="ratio-y"]`);
		this.handleButtons = {
			topLeft: this.window.querySelector(".crop-guideline-top-left"),
			topRight: this.window.querySelector(".crop-guideline-top-right"),
			bottomRight: this.window.querySelector(".crop-guideline-bottom-right"),
			bottomLeft: this.window.querySelector(".crop-guideline-bottom-left")
		};

		/**
		 * The original width of the image being cropped
		 * @type {?number}
		 */
		this.imageOriginalWidth = null;

		/**
		 * The original height of the image being cropped
		 * @type {?number}
		 */
		this.imageOriginalHeight = null;

		/**
		 * If the mouse is currently held down on a handle button
		 * @type {boolean}
		 */
		this.isMouseDownOnHandleButton = false;

		/**
		 * If the mouse is currently held down on the inner part of the
		 * grid guidelines container and not a button. Used to move the guidelines container
		 * itself.
		 * @type {boolean}
		 */
		this.isMouseDownOnGuidelinesContainer = false;

		/**
		 * The current handle button being dragged
		 * @type {HTMLButtonElement}
		 */
		this.currentlyDraggedHandleButton = null;

		/**
		 * The last mouse position recorded. This is populated upon mouse down
		 * on a handle button or the guidelines container
		 */
		this.lastRecordedMousePosition = {
			x:null,
			y:null
		};

		/** @type {boolean} Flag for when the form is submitting and being processed */
		this.isProcessing = false;

		/**
		 * The maximum size, in pixels, the grid guidelines container can be.
		 * These will be updated each time a new image is loaded into the crop window
		 * @type {{width: number, height: number}}
		 */
		this.guidelinesSizeMaximums = {
			width:parseInt(window.getComputedStyle(this.guidelinesBackground).width),
			height:parseInt(window.getComputedStyle(this.guidelinesBackground).height)
		};

		/**
		 * The crop guideline sizes in percentage. Defaults to 100% because that
		 * is what is defined in the stylesheet
		 * @type {{width: number, height: number}}
		 */
		this.guidelinesSizePercentages = {
			width:100,
			height:100
		};

		for (const handleButton of Object.values(this.handleButtons)){
			handleButton.addEventListener("mousedown", e => {
				this.onHandleButtonMouseDown(handleButton, e);
			});
		}

		this.ratioXInput.addEventListener("input", () => {
			this.onAspectInputChanged(this.ratioXInput);
		});

		this.ratioYInput.addEventListener("input", () => {
			this.onAspectInputChanged(this.ratioYInput);
		});

		this.formSubmitButton.addEventListener("click", () => {
			this.onSubmit();
		});

		this.form.addEventListener("submit",e => {
			e.preventDefault();
			this.onSubmit();
		});

		this.guidelinesContainer.addEventListener("mousedown", (/** @type {MouseEvent} */ e) => {
			// Is it a div inside the grid guidelines grid elements?
			if (e.target.parentElement.closest(".crop-guidelines-grid") !== null){
				this.lastRecordedMousePosition.x = e.pageX;
				this.lastRecordedMousePosition.y = e.pageY;
				this.isMouseDownOnGuidelinesContainer = true;
			}
		});

		this.dismissButton.addEventListener("click", () => {
			if (!this.isProcessing){
				this.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);
			}
		});

		// Event to reset mouse down flags
		window.addEventListener("mouseup", () => {
			if (this.isMouseDownOnHandleButton){
				this.isMouseDownOnHandleButton = false;
			}

			if (this.isMouseDownOnGuidelinesContainer){
				this.isMouseDownOnGuidelinesContainer = false;
			}
		});

		window.addEventListener("mousemove", e => {
			if (this.isMouseDownOnHandleButton) {
				this.onHandleButtonDragged(this.currentlyDraggedHandleButton, e);
			}else if (this.isMouseDownOnGuidelinesContainer){
				this.onGuidelinesContainerDragged(e);
			}
		});
	}

	/**
	 * When the crop form is submitted
	 */
	async onSubmit(){
		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;

		// Properly fill in the input fields for the topX, topY, bottomX, bottomY
		// corner locations

		// Fetch the computer style of the preview image to determine
		// the relative crop location if the image's actual dimensions are larger
		// than what can be shown in the preview
		const computedStylesForPreviewImage = window.getComputedStyle(this.previewImage);
		const computedStylesForGuidelines = window.getComputedStyle(this.guidelinesContainer);
		const previewImageWidth = parseFloat(computedStylesForPreviewImage.width);
		const previewImageHeight = parseFloat(computedStylesForPreviewImage.height);
		const guidelineWidth = parseFloat(computedStylesForGuidelines.width);
		const guidelineHeight = parseFloat(computedStylesForGuidelines.height);

		const topX = parseFloat(computedStylesForGuidelines.left);
		const topY = parseFloat(computedStylesForGuidelines.top);
		const topXPercentage = topX / previewImageWidth;
		const topYPercentage = topY / previewImageHeight;
		const bottomXPercentage = (topX + guidelineWidth) / previewImageWidth;
		const bottomYPercentage = (topY + guidelineHeight) / previewImageHeight;

		// Get the locations of the topX,topY,bottomX,bottomY on the image original size
		const realTopX = topXPercentage * this.imageOriginalWidth;
		const realTopY = topYPercentage * this.imageOriginalHeight;
		const realBottomX = bottomXPercentage * this.imageOriginalWidth;
		const realBottomY = bottomYPercentage * this.imageOriginalHeight;

		this.previewContainer.style.display = "none";
		this.spinner.style.display = "block";

		const fData = new FormData(this.form);
		fData.set("top-x", String(Math.round(realTopX)));
		fData.set("top-y", String(Math.round(realTopY)));
		fData.set("bottom-x", String(Math.round(realBottomX)));
		fData.set("bottom-y", String(Math.round(realBottomY)));

		const response = await fetch(`/uplift/image-manager/crop-image`, {
			credentials:"same-origin",
			cache:"no-cache",
			body:fData,
			method:"patch"
		});

		if (response.status === 200){
			const data = await response.json();
			if (data.status === 1){
				// Refresh the images directory to reflect the new data on the resized image
				const imageDirectoryFetcher = this.imageManager.directoryImagesFetcher;
				imageDirectoryFetcher.setDirectory(this.imageManager.imageManagerState.currentDirectory);
				await imageDirectoryFetcher.fetchAndRender();

				this.imageManager.metaDataPane.hidePreview();
				this.imageManager.metaDataPane.hideImageData();
				this.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);

				// Refresh the cache string that is appended to thumbs in the sidebar preview
				this.imageManager.imageManagerState.currentImageCacheRandomString = String(Math.random());
			}else if (data.status === -1){
				alert(data.error);
			}
		}else{
			alert("Non-200 OK error. Check the network tab.");
		}

		this.isProcessing = false;
		this.spinner.style.display = "none";
		this.previewContainer.style.display = "inline-block";
	}

	/**
	 * Function called when this window gets shown
	 * @param {string} imageFilePath
	 * @param {string} imageURI
	 * @param {number} imageOriginalWidth
	 * @param {number} imageOriginalHeight
	 */
	onWindowShown(imageFilePath, imageURI, imageOriginalWidth, imageOriginalHeight){
		this.imageFilePathInput.value = imageFilePath;
		this.imageOriginalWidth = imageOriginalWidth;
		this.imageOriginalHeight = imageOriginalHeight;

		const newImageObj = new Image();
		const randomString = Math.random();

		// Wait for the image to load before updating the crop guidelines
		// Otherwise it will not detect the right height and width of the container
		newImageObj.onload = async () => {
			this.previewImage.setAttribute("src", imageURI + `?no-cache=${randomString}`);
			await Scheduler.wait(50);

			// Update the maximums
			this.guidelinesSizeMaximums = {
				width:parseInt(window.getComputedStyle(this.guidelinesBackground).width),
				height:parseInt(window.getComputedStyle(this.guidelinesBackground).height)
			};

			this.updateCropGuideToMatchAspectRatio();
		};

		newImageObj.onerror = () => {

		};

		newImageObj.src = imageURI + `?no-cache=${randomString}`;
	}

	/**
	 * When the mouse has dragged a handle button while holding the LMB down
	 * @param {HTMLButtonElement} handleButton
	 * @param {MouseEvent} event
	 */
	onHandleButtonDragged(handleButton, event){
		const movementX = event.pageX - this.lastRecordedMousePosition.x;
		const movementY = event.pageY - this.lastRecordedMousePosition.y;

		const aspectRatio = Number(this.ratioYInput.value / this.ratioXInput.value);
		const computedStyles = window.getComputedStyle(this.guidelinesContainer);
		const currentWidth = parseFloat(computedStyles.width);
		const currentHeight = parseFloat(computedStyles.height);
		const currentLeft = parseFloat(computedStyles.left);
		const currentTop = parseFloat(computedStyles.top);

		let newWidth = currentWidth;
		let newLeft = currentLeft;
		let newHeight = currentHeight;
		let newTop = currentTop;

		// Move the top/left properties and resize the guidelines container element
		// based on which handle was dragged and in what direction
		if (handleButton === this.handleButtons.bottomLeft){
			newWidth = currentWidth - movementX;
			newHeight = currentHeight + movementY;
			let additionalXMovement = 0;

			if (!isNaN(aspectRatio)){
				if (Math.abs(movementY) > Math.abs(movementX)){
					// Y axis is dominant. Determine new width
					// Find the width that is closest in aspect ratio to the new height
					const newAspectRatioWidth = newHeight / aspectRatio;
					additionalXMovement += (newAspectRatioWidth - newWidth);
					newWidth = newAspectRatioWidth;
				}else{
					newHeight = newWidth * aspectRatio;
				}
			}

			newLeft = currentLeft + movementX - additionalXMovement;
		}else if (handleButton === this.handleButtons.bottomRight){
			newWidth = currentWidth + movementX;
			newHeight = currentHeight + movementY;

			if (!isNaN(aspectRatio)) {
				if (Math.abs(movementY) > Math.abs(movementX)) {
					newWidth = newHeight / aspectRatio;
				}else{
					newHeight = newWidth * aspectRatio;
				}
			}
		}else if (handleButton === this.handleButtons.topLeft){
			newWidth = currentWidth - movementX;
			newHeight = currentHeight - movementY;
			let additionalXMovement = 0;
			let additionalYMovement = 0;

			if (!isNaN(aspectRatio)){
				if (Math.abs(movementY) > Math.abs(movementX)){
					// Y axis is dominant. Determine new width
					// Find the width that is closest in aspect ratio to the new height
					const newAspectRatioWidth = newHeight / aspectRatio;
					additionalXMovement += (newAspectRatioWidth - newWidth);
					newWidth = newAspectRatioWidth;
				}else{
					const newAspectRatioHeight = newWidth * aspectRatio;
					additionalYMovement += (newAspectRatioHeight - newHeight);
					newHeight = newAspectRatioHeight;
				}
			}

			newLeft = currentLeft + movementX - additionalXMovement;
			newTop = currentTop + movementY - additionalYMovement;
		}else if (handleButton === this.handleButtons.topRight){
			newWidth = currentWidth + movementX;
			newHeight = currentHeight - movementY;
			let additionalYMovement = 0;

			if (!isNaN(aspectRatio)){
				if (Math.abs(movementY) > Math.abs(movementX)){
					// Y axis is dominant. Determine new width
					// Find the width that is closest in aspect ratio to the new height
					newWidth = newHeight / aspectRatio;
				}else{
					const newAspectRatioHeight = newWidth * aspectRatio;
					additionalYMovement += (newAspectRatioHeight - newHeight);
					newHeight = newAspectRatioHeight;
				}
			}

			newTop = currentTop + movementY - additionalYMovement;
		}

		if (newLeft + newWidth > this.guidelinesSizeMaximums.width){
			// Right side overflow
			newWidth = (this.guidelinesSizeMaximums.width - newLeft)
			newHeight = currentHeight;
			newTop = currentTop;
		}

		if (newTop + newHeight > this.guidelinesSizeMaximums.height){
			// Bottom side overflow
			newHeight = (this.guidelinesSizeMaximums.height - newTop)
			newWidth = currentWidth;
			newLeft = currentLeft;
		}

		if (newLeft < 0){
			// Left side overflow
			newLeft = 0;
			newTop = currentTop;
			newWidth = currentWidth;
			newHeight = currentHeight;
		}

		if (newTop < 0){
			// Top side overflow
			newTop = 0;
			newLeft = currentLeft;
			newWidth = currentWidth;
			newHeight = currentHeight;
		}

		this.guidelinesContainer.style.width = `${newWidth}px`;
		this.guidelinesContainer.style.height = `${newHeight}px`;
		this.guidelinesContainer.style.left = `${newLeft}px`;
		this.guidelinesContainer.style.top = `${newTop}px`;

		// Update the last recorded mouse position
		this.lastRecordedMousePosition.x = event.pageX;
		this.lastRecordedMousePosition.y = event.pageY;

		// this.checkAndHandleGuidelineOverflow();
	}

	/**
	 * Checks for overflow and corrects the overflow of the grid guidelines
	 */
	checkAndHandleGuidelineOverflow(){
		const computedStyles = window.getComputedStyle(this.guidelinesContainer);
		// Check for overflow
		// Check if the top-left corner is overflowing
		const topLeftCornerPosition = {
			x:parseFloat(computedStyles.left),
			y:parseFloat(computedStyles.top)
		};

		const bottomRightCornerPosition = {
			x:topLeftCornerPosition.x + parseFloat(computedStyles.width),
			y:topLeftCornerPosition.y + parseFloat(computedStyles.height)
		};

		// Left side overflow
		if (topLeftCornerPosition.x < 0){
			// Correct the left side overflow
			const difference = Math.abs(topLeftCornerPosition.x);
			this.guidelinesContainer.style.left = "0";
			this.guidelinesContainer.style.width = `${parseFloat(computedStyles.width) + difference}px`;
		}

		// Top side overflow
		if (topLeftCornerPosition.y < 0){
			const difference = Math.abs(topLeftCornerPosition.y);
			this.guidelinesContainer.style.top = "0";
			this.guidelinesContainer.style.height = `${parseFloat(computedStyles.height) + difference}px`;
		}

		// Right side overflow
		if (bottomRightCornerPosition.x > this.guidelinesSizeMaximums.width){
			if (parseFloat(computedStyles.width) === this.guidelinesSizeMaximums.width){
				this.guidelinesContainer.style.left = "0";
				this.guidelinesContainer.style.width = `${this.guidelinesSizeMaximums.width}px`;
			}else {
				const difference = Math.abs(bottomRightCornerPosition.x - this.guidelinesSizeMaximums.width);
				this.guidelinesContainer.style.left = `${parseFloat(computedStyles.left) - difference}px`;
				this.guidelinesContainer.style.width = `${parseFloat(computedStyles.width) - difference}px`;
			}
		}

		// Bottom side overflow
		if (bottomRightCornerPosition.y > this.guidelinesSizeMaximums.height){
			if (parseFloat(computedStyles.height) === this.guidelinesSizeMaximums.height) {
				this.guidelinesContainer.style.top = "0";
				this.guidelinesContainer.style.height = `${this.guidelinesSizeMaximums.height}px`;
			}else{
				const difference = Math.abs(bottomRightCornerPosition.y - this.guidelinesSizeMaximums.height);
				this.guidelinesContainer.style.top = `${parseFloat(computedStyles.top) - difference}px`;
				this.guidelinesContainer.style.height = `${parseFloat(computedStyles.height) - difference}px`;
			}
		}
	}

	/**
	 *
	 * @param {HTMLButtonElement} handleButton
	 * @param {MouseEvent} event
	 */
	onHandleButtonMouseDown(handleButton, event){
		this.isMouseDownOnHandleButton = true;
		this.currentlyDraggedHandleButton = handleButton;
		this.lastRecordedMousePosition.x = event.pageX;
		this.lastRecordedMousePosition.y = event.pageY;
	}

	/**
	 * Updates the crop grid outline to match the aspect ratio set by the inputs
	 */
	updateCropGuideToMatchAspectRatio(){
		const ratioX = parseInt(this.ratioXInput.value);
		const ratioY = parseInt(this.ratioYInput.value);

		// Get the maximum height that can be considered for the grid outline.
		// This is based on where the box was last manually dragged
		const maxWidthToConsider = this.guidelinesSizeMaximums.width * (this.guidelinesSizePercentages.width / 100);
		const maxHeightToConsider = this.guidelinesSizeMaximums.height * (this.guidelinesSizePercentages.width / 100);

		if (!isNaN(ratioX) && !isNaN(ratioY)){
			const currentAspectRatioRounded = (maxHeightToConsider / maxWidthToConsider).toFixed(2);
			const desiredAspectRatioRounded = (ratioY / ratioX).toFixed(2);
			if (currentAspectRatioRounded !== desiredAspectRatioRounded) {
				const newHeight = desiredAspectRatioRounded * maxWidthToConsider;
				if (newHeight > maxHeightToConsider) {
					// Adjust the width instead
					const newWidth = maxHeightToConsider / desiredAspectRatioRounded;
					this.guidelinesContainer.style.width = `${newWidth}px`;
					this.guidelinesContainer.style.height = null;
				} else {
					this.guidelinesContainer.style.height = `${newHeight}px`;
					this.guidelinesContainer.style.width = null;
				}
			}
		}
	}

	/**
	 * When any aspect ratio input boxes are changed
	 * @param {HTMLInputElement} inputElementChanged
	 */
	onAspectInputChanged(inputElementChanged){
		this.updateCropGuideToMatchAspectRatio();
	}

	/**
	 * When the guidelines container itself is dragged.
	 * @param {MouseEvent} event
	 */
	onGuidelinesContainerDragged(event){
		const movementX = event.pageX - this.lastRecordedMousePosition.x;
		const movementY = event.pageY - this.lastRecordedMousePosition.y;

		const computedStyles = window.getComputedStyle(this.guidelinesContainer);
		const currentLeft = parseFloat(computedStyles.left);
		const currentTop = parseFloat(computedStyles.top);
		const width = parseFloat(computedStyles.width);
		const height = parseFloat(computedStyles.height);
		let newLeft = currentLeft + movementX;
		let newTop = currentTop + movementY;

		if (newLeft < 0){
			newLeft = 0;
		}

		if (newTop < 0){
			newTop = 0;
		}

		if (width + newLeft > this.guidelinesSizeMaximums.width){
			newLeft -= ((newLeft + width) - this.guidelinesSizeMaximums.width);
		}

		if (height + newTop > this.guidelinesSizeMaximums.height){
			newTop -= ((newTop + height) - this.guidelinesSizeMaximums.height);
		}

		this.guidelinesContainer.style.left = `${newLeft}px`;
		this.guidelinesContainer.style.top = `${newTop}px`;

		// Update the last recorded mouse position
		this.lastRecordedMousePosition.x = event.pageX;
		this.lastRecordedMousePosition.y = event.pageY;
	}
}

export default CropWindow;