import SelectionListener from "../SelectionListener.js";
import WindowManager from "../WindowManager.js";
import ImageManager from "../ImageManager.js";
import EditorWrapper from "../../editors/page-editor/EditorWrapper.js";

class QuickGalleryForm{
	/**
	 * @type {ImageManager}
	 */
	imageManager = null;
	isProcessing = false;
	form = document.querySelector("#im-quick-gallery-form");
	loader = document.querySelector("#im-quick-gallery-spinner");

	/**
	 * Set the "Make Quick Gallery" button is clicked. Will be set to the last selection of image components used.
	 * @type {array|null}
	 */
	imageComponentsToUse = null;

	constructor(imageManager) {
		this.imageManager = imageManager;
		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	 * @returns {Promise<Object>}
	 */
	async onSubmit(){
		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;
		this.form.style.display = "none";
		this.loader.style.display = null;

		const fData = new FormData(this.form);

		const response = await fetch(`/uplift/gallery-manager/gallery`, {
		    body:fData,
		    method:"PUT",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		let data;
		try{
		    /** @type {{status: int, error: ?string, newGallery: Object}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
			this.isProcessing = false;
			alert(jsonSyntaxError);
			return;
		}

		if (data.status === 1){
			/** @type {{id: int, name: string}} */
			const newGallery = data.newGallery;

			const payload = {images: []};
			for (const [index, imageComponent] of this.imageComponentsToUse.entries()){
				// Add these to the gallery
				payload.images.push({
					position: index,
					imageURI: imageComponent.uri,
					thumbURI: imageComponent.thumbURI,
					altText: imageComponent.tryToCreateAlt()
				})
			}

			const headers = new Headers();
			headers.set("content-type", "application/json; charset=utf-8");

			await fetch(`/uplift/gallery-manager/gallery/${newGallery.id}/members`, {
				method: "PUT",
				body:JSON.stringify(payload),
				cache: "no-cache",
				credentials: "same-origin",
				headers: headers
			});

			if (this.imageManager.currentAceEditor !== null){
				let galleryName_sanitized = newGallery.name;

				// Remove double quotes, {, and } characters
				galleryName_sanitized = galleryName_sanitized.replace("\"", "");
				galleryName_sanitized = galleryName_sanitized.replace("{", "");
				galleryName_sanitized = galleryName_sanitized.replace("}", "");

				const shortcode = `{{ gallery id="${newGallery.id}" gallery-name="${galleryName_sanitized}" }}`;
				this.imageManager.currentAceEditor.session.replace(this.imageManager.currentAceEditor.session.getSelection().getRange(), shortcode);
			}else{
				console.log("ImageManager.currentAceEditor was not set when the ImageManager was opened. Cannot inject gallery shortcode.");
			}

			// Fire the promise that editors may be waiting on so future image manager opens don't duplicate results
			this.imageManager.fireImagesInserted([]);
			this.imageManager.windowManager.closeImageManager();
		}else if (data.status === -1){
			alert(data.error);
			console.warn(data.error);
		}

		this.isProcessing = false;
		this.form.style.display = null;
		this.loader.style.display = "none";
	}


}

export default QuickGalleryForm;