import ImageManagerState from "./ImageManagerState.js";
import ImageComponent from "./components/ImageComponent.js";
import ContextMenu from "../utils/ContextMenu.js";

class SelectionListener{

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @param {ImageManager} imageManager
	 */
	constructor(imageManager){
		this.imageManager = imageManager;

		this.footerButtons = this.imageManager.container.querySelector(".im-footer-buttons");

		/** @type {ImageComponent[]} currentImageComponentSelection */
		this.currentImageComponentSelection = [];

		/** @type {Promise} imageChosenPromise */
		this.imageChosenPromise = null;

		/** @type {function} imageChosenPromiseAcceptFunction */
		this.imageChosenPromiseAcceptFunction = null;

		/** @type {HTMLElement} nextFocusElement The next element that will be receiving focus */
		this.nextFocusElement = null;

		/** @type {HTMLElement} currentFocusElement The current element in focus */
		this.currentFocusElement = null;

		this.imageManager.container.addEventListener("focusin", e => {
			/** @type {?HTMLElement} */
			const target = e.target;
			if (target !== null){
				this.nextFocusElement = target;
				this.onSelectionChanged();
			}
		});

		this.imageManager.container.addEventListener("focusout", e => {
			this.currentFocusElement = null;
			setTimeout(() => {
				if (!this.currentFocusElement && ContextMenu.currentContextMenu === null){
					this.onSelectionNull();
				}
			}, 50);
		});

		window.addEventListener("keydown", e => {
			const currentlyFocusedImageComponent = this.getCurrentlyFocusedImageComponent();
			if (currentlyFocusedImageComponent) {
				if (e.code === "ArrowDown") {
					this.adjustCurrentImageComponentSelection(currentlyFocusedImageComponent, "down", e);
				}else if (e.code === "ArrowUp"){
					this.adjustCurrentImageComponentSelection(currentlyFocusedImageComponent, "up", e);
				}
			}
		});
	}

	/**
	 * Adjusts the current selection (or even clears it I guess if no shift key)
	 * based on the arrow key direction
	 * @param {ImageComponent} component
	 * @param {string} direction
	 * @param {KeyboardEvent} e
	 */
	adjustCurrentImageComponentSelection(component, direction, e){
		e.preventDefault();
		if (direction === "up"){
			// Select the previous sibling
			const previousSibling = component.dom.previousElementSibling;
			if (previousSibling){
				previousSibling.focus();
				const previousComponent = ImageComponent.getComponentByDOM(previousSibling);
				if (e.shiftKey) {
					this.addImageComponentsToSelectionFromRange(
						this.currentImageComponentSelection[0],
						previousComponent
					);
				}else{
					// Clear selection and add the new component
					this.clearImageComponentSelection();
					this.addImageComponentToSelection(previousComponent);
				}
			}
		}else if (direction === "down"){
			// Select the next sibling
			const nextSibling = component.dom.nextElementSibling;
			if (nextSibling){
				nextSibling.focus();
				const nextComponent = ImageComponent.getComponentByDOM(nextSibling);
				if (e.shiftKey) {
					this.addImageComponentsToSelectionFromRange(
						this.currentImageComponentSelection[0],
						nextComponent
					);
				}else{
					// Clear selection and add the new component
					this.clearImageComponentSelection();
					this.addImageComponentToSelection(nextComponent);
				}
			}
		}
	}

	/**
	 * Checks if the document's active/focused element is an ImageComponent
	 * @return {?ImageComponent}
	 */
	getCurrentlyFocusedImageComponent(){
		const focusedElement = document.activeElement;
		return ImageComponent.getComponentByDOM(focusedElement);
	}

	/**
	 * @return {boolean}
	 */
	hasImageComponentsSelected(){
		return this.currentImageComponentSelection.length > 0;
	}

	/**
	 * Fetches the index of a component in the current selection
	 * @param {ImageComponent} component
	 * @return {?int}
	 */
	getImageComponentArrayIndexInSelectionArray(component){
		const hashID = parseInt(component.dom.getAttribute("hash-lookup-id"));
		for (/** @type {int} */ const index in this.currentImageComponentSelection){
			const existingComponent = this.currentImageComponentSelection[index];
			const thisHashID = parseInt(existingComponent.dom.getAttribute("hash-lookup-id"));
			if (thisHashID === hashID){
				return index;
			}
		}

		return null;
	}

	/**
	 * Checks if an image component is in the selection
	 * @param {ImageComponent} component
	 * @return {boolean}
	 */
	isImageComponentInSelection(component){
		return this.getImageComponentArrayIndexInSelectionArray(component) !== null;
	}

	/**
	 * Adds an ImageComponent to the selection
	 * @param {ImageComponent} component
	 */
	addImageComponentToSelection(component){
		component.dom.classList.add("in-selection");
		this.currentImageComponentSelection.push(component);

		if (this.currentImageComponentSelection.length === 1){
			// Only one image is selected, show the URI
			this.imageManager.container.querySelector(".im-selected-images-footer").style.display = "none";
			this.imageManager.container.querySelector(".im-selected-image-footer").style.display = "block";
			this.imageManager.container.querySelector(".im-quick-gallery-button").style.display = "none";
			this.imageManager.container.querySelector(".im-selected-image-uri").textContent = component.uri;
		}else{
			if (this.currentImageComponentSelection.length > 1){
				this.imageManager.container.querySelector(".im-selected-images-footer").style.display = "block";
				this.imageManager.container.querySelector(".im-selected-image-footer").style.display = "none";
				this.imageManager.container.querySelector(".im-quick-gallery-button").style.display = null;
				this.imageManager.container.querySelector(".im-selected-images-amount").textContent = String(this.currentImageComponentSelection.length);
			}else{
				// It's 0
				this.imageManager.container.querySelector(".im-selected-images-footer").style.display = "none";
				this.imageManager.container.querySelector(".im-selected-image-footer").style.display = "none";
				this.imageManager.container.querySelector(".im-quick-gallery-button").style.display = "none";
			}
		}
	}

	/**
	 * Removes an ImageComponent from the selection
	 * @param {ImageComponent} component
	 */
	removeImageComponentFromSelection(component){
		component.dom.classList.remove("in-selection");
		const arrayIndex = this.getImageComponentArrayIndexInSelectionArray(component);
		this.currentImageComponentSelection.splice(arrayIndex, 1);
	}

	/**
	 * Selects all image components starting from a providing component
	 * and chronologically to the end component
	 * @param {ImageComponent} startComponent
	 * @param {ImageComponent} endComponent
	 */
	addImageComponentsToSelectionFromRange(startComponent, endComponent){

		if (startComponent !== endComponent) {
			let startNodePosition = 0;
			let endNodePosition = 0;
			const startDOM = startComponent.dom;
			const endDOM = endComponent.dom;

			// Clear the current selection
			this.clearImageComponentSelection();

			// Determine which direction to go

			// Get the node position of the start
			let startPrevSibling = startDOM.previousElementSibling;
			while(startPrevSibling !== null){
				++startNodePosition;
				startPrevSibling = startPrevSibling.previousElementSibling;
			}

			// Get the node position of the end
			let endPrevSibling = endDOM.previousElementSibling;
			while(endPrevSibling !== null){
				++endNodePosition;
				endPrevSibling = endPrevSibling.previousElementSibling;
			}

			if (startNodePosition < endNodePosition){
				// Select all nodes below startDOM
				for (let i = startNodePosition; i <= endNodePosition; i++){
					const imImageNode = startDOM.parentElement.children[i];
					const componentForNode = ImageComponent.getComponentByDOM(imImageNode);
					this.addImageComponentToSelection(componentForNode);
				}
			}else{
				// Select all nodes above startDOM
				for (let i = startNodePosition; i >= endNodePosition; i--){
					const imImageNode = startDOM.parentElement.children[i];
					const componentForNode = ImageComponent.getComponentByDOM(imImageNode);
					this.addImageComponentToSelection(componentForNode);
				}
			}

		}else{
			this.clearImageComponentSelection();
			this.addImageComponentToSelection(startComponent);
		}
	}

	/**
	 * Clears the current selection of image components
	 */
	clearImageComponentSelection(){
		for (/** @type {ImageComponent} */ const component of this.currentImageComponentSelection){
			component.dom.classList.remove("in-selection");
		}

		this.currentImageComponentSelection = [];
		this.imageManager.container.querySelector(".im-selected-images-footer").style.display = "none";
		this.imageManager.container.querySelector(".im-selected-image-footer").style.display = "none";
	}

	async imageChosen(){
		return this.imageChosenPromise;
	}

	/**
	 * When the selection inside the image manager changes
	 */
	onSelectionChanged(){
		if (this.nextFocusElement.closest("im-image")){
			this.currentFocusElement = this.nextFocusElement;
			this.onImageSelected(this.currentFocusElement.closest("im-image"));
		}else if (this.nextFocusElement.closest(("im-window-footer"))){
			this.currentFocusElement = this.nextFocusElement;
			this.onFooterButtonSelected(this.currentFocusElement.closest("im-window-footer"));
		}else{
			// No known element
			this.currentFocusElement = null;
		}

		this.nextFocusElement = null;
	}

	/**
	 * When an image item (im-image element) is selected
	 * @param {HTMLElement} element The im-image element
	 */
	onImageSelected(element){
		this.footerButtons.style.display = "block";

		if (this.imageManager.imageManagerState.imageMode === this.imageManager.imageManagerState.IMAGE_MODES.INSERT){
			this.imageManager.container.querySelector(".im-insertion-mode-buttons").style.display = "flex";
			this.imageManager.container.querySelector(".im-selection-mode-buttons").style.display = "none";
		}else if (this.imageManager.imageManagerState.imageMode === this.imageManager.imageManagerState.IMAGE_MODES.SELECT){
			this.imageManager.container.querySelector(".im-insertion-mode-buttons").style.display = "none";
			this.imageManager.container.querySelector(".im-selection-mode-buttons").style.display = "flex";
		}
	}

	/**
	 * When a button in the image manager footer is selected
	 * @param {HTMLElement} element
	 */
	onFooterButtonSelected(element){
		this.footerButtons.style.display = "block";
	}

	/**
	 * When the selection is changed to null
	 */
	onSelectionNull(){
		this.footerButtons.style.display = "none";
		this.clearImageComponentSelection();
	}
}

export default SelectionListener;