import DirectoriesFetcher from "./DirectoriesFetcher.js";
import DirectoryImagesFetcher from "./DirectoryImagesFetcher.js";
import ImageManagerState from "./ImageManagerState.js";
import FilterImages from "./components/FilterImages.js";
import SelectionListener from "./SelectionListener.js";
import WindowManager from "./WindowManager.js";
import FoldersSidebar from "./components/FoldersSidebar.js";
import OSFileUploadComponent from "./components/OSFileUploadComponent.js";
import InsertImagesButton from "./components/InsertImagesButton.js";
import SelectImagesButton from "./components/SelectImagesButton.js";
import MakeQuickGalleryButton from "./components/MakeQuickGalleryButton.js";
import QuickGalleryForm from "./forms/QuickGalleryForm.js";
import RefreshDirectoriesButton from "./components/RefreshDirectoriesButton.js";
import BreadcrumbHandler from "./BreadcrumbHandler.js";
import MetaDataPane from "./MetaDataPane.js";
import Scheduler from "../utils/Scheduler.js";

class ImageManager{

	/**
	 * @type {HTMLElement}
	 */
	container;

	/**
	 * @type {?DirectoriesFetcher}
	 */
	directoriesFetcher = null;

	/**
	 * @type {?DirectoryImagesFetcher}
	 */
	directoryImagesFetcher = null;

	/**
	 * @type {?ImageManagerState}
	 */
	imageManagerState = null;

	/**
	 * @type {?FilterImages}
	 */
	filterImages = null;

	/**
	 * @type {?WindowManager}
	 */
	windowManager = null;

	/**
	 * @type {?FoldersSidebar}
	 */
	foldersSidebar = null;

	/**
	 * @type {?OSFileUploadComponent}
	 */
	osFileUploadComponent = null;

	/**
	 * @type {?InsertImagesButton}
	 */
	insertImagesButton = null;

	/**
	 * @type {?SelectImagesButton}
	 */
	selectImagesButton = null;

	/**
	 * @type {?MakeQuickGalleryButton}
	 */
	makeQuickGalleryButton = null;

	/**
	 * @type {?QuickGalleryForm}
	 */
	quickGalleryForm = null;

	/**
	 * @type {?RefreshDirectoriesButton}
	 */
	refreshDirectoriesButton = null;

	/**
	 * @type {?BreadcrumbHandler}
	 */
	breadcrumbsHandler = null;

	/**
	 * @type {?SelectionListener}
	 */
	selectionListener = null;

	/**
	 * @type {?MetaDataPane}
	 */
	metaDataPane = null;

	currentAceEditor = null;

	/**
	 * @param {HTMLElement} container
	 */
	constructor(container){

		this.container = container;
		this.imagesChosen = null;
		this.imagesChosenPromiseResolveFunction = null;

		// Initialize
		this.init();
	}

	/**
	 * @param {Object} aceEditor
	 */
	setCurrentAceEditor(aceEditor){
		this.currentAceEditor = aceEditor;
	}

	/**
	 * This function exists because a constructor() cannot be async so dumb omg
	 * @returns {Promise<void>}
	 */
	async init(){

		// Initialize instances of necessary classes
		this.directoriesFetcher = new DirectoriesFetcher(this);
		this.directoryImagesFetcher = new DirectoryImagesFetcher(this, "");
		this.imageManagerState = new ImageManagerState(this);
		this.filterImages = new FilterImages(this);
		this.windowManager = new WindowManager(this);
		this.foldersSidebar = new FoldersSidebar(this);
		this.osFileUploadComponent = new OSFileUploadComponent(this);
		this.insertImagesButton = new InsertImagesButton(this);
		this.selectImagesButton = new SelectImagesButton(this);
		this.makeQuickGalleryButton = new MakeQuickGalleryButton(this);
		this.quickGalleryForm = new QuickGalleryForm(this);
		this.refreshDirectoriesButton = new RefreshDirectoriesButton(this);
		this.breadcrumbsHandler = new BreadcrumbHandler(this);
		this.selectionListener = new SelectionListener(this);
		this.metaDataPane = new MetaDataPane(this);

		await this.loadDirectories();
		await this.loadImagesInDirectory();
	}

	async loadDirectories(directory){
		if (!this.directoriesFetcher.isBusy){
			await this.directoriesFetcher.fetchAndRender(directory);
		}
	}

	/**
	 * Loads the images in the current directory. The current directory is fetched from the imageManagerState.currentDirectory
	 */
	async loadImagesInDirectory(){
		const directoryImagesFetcher = this.directoryImagesFetcher;
		directoryImagesFetcher.setDirectory(this.imageManagerState.currentDirectory);
		await directoryImagesFetcher.fetchAndRender();
	}

	/**
	 * @returns {Element}
	 */
	getSidebarFoldersContainer(){
		return this.container.querySelector("im-window-body .folder-column > .folders-container");
	}

	getImageComponentsContainer(){
		return this.container.querySelector("image-manager-container .im-images");
	}

	getBreadcrumbsContainer(){
		return this.container.querySelector("im-location-bar .location-buttons");
	}

	/**
	 * Fires the images inserted callbacks
	 * @param {ImageComponent[]} imageComponents
	 */
	fireImagesInserted(imageComponents){
		this.imagesChosenPromiseResolveFunction(imageComponents);
	}

	/**
	 * Fires the images selected callbacks
	 * @param {ImageComponent[]} imageComponents
	 */
	fireImagesSelected(imageComponents){
		this.imagesChosenPromiseResolveFunction(imageComponents);
	}
}

export default ImageManager;