import ImageManagerState from "./ImageManagerState.js";
import DirectoryImagesFetcher from "./DirectoryImagesFetcher.js";

class BreadcrumbHandler{

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @type {HTMLElement}
	 */
	container;

	/**
	 * @param {ImageManager} imageManager
	 */
	constructor(imageManager){
		this.imageManager = imageManager;
		this.container = imageManager.getBreadcrumbsContainer();
		this.currentDOMs = [];
	}

	/**
	 * Fetches a location crumb
	 * @param {string} crumbLabel
	 * @param {string} fullDirectoryPath
	 */
	getCrumbDOM(crumbLabel, fullDirectoryPath){
		const template = document.createElement("im-location-button");
		template.setAttribute("filesystem-path", fullDirectoryPath);
		template.innerHTML = `
			<span class=\"folder-name\">${crumbLabel}</span>
		`;

		template.addEventListener("click", () => {
			const imFetcher = this.imageManager.directoryImagesFetcher;
			imFetcher.setDirectory(fullDirectoryPath);
			imFetcher.fetchAndRender();
		});

		this.currentDOMs.push(template);

		return template;
	}

	/**
	 * Breaks the current directory from the image manager state into an array of
	 * crumb paths.
	 * @return {array}
	 */
	getCurrentDirectoryAsArrayPath(){
		const baseDirectory = this.imageManager.imageManagerState.baseImagesDirectory;
		const baseNameRegex = /[\/\\]([^\/\\]+?)$/;
		let currentDirectoryPath = this.imageManager.imageManagerState.currentDirectory.replace(baseDirectory, "");
		let loops = 0;
		const locations = [];

		while (currentDirectoryPath !== "" && loops < 20){
			const baseNameMatches = currentDirectoryPath.match(baseNameRegex);

			// Handle failure. Usually happens in local environments due to the C: or HDD path in general
			if (baseNameMatches === null){
				console.warn(`Directory regex failed. Finally directory processed that caused the match failure is ${currentDirectoryPath}`);
				break;
			}

			const dirBaseName = baseNameMatches[1];
			locations.push({
				"fullDirectory":`${baseDirectory}${currentDirectoryPath}`,
				"baseName":dirBaseName
			});
			currentDirectoryPath = currentDirectoryPath.replace(/[\/\\][^\/\\]+?$/, "");
			++loops;
		}

		locations.push({
			"fullDirectory":baseDirectory,
			"baseName":(baseDirectory.match(baseNameRegex))[1]
		});

		locations.reverse();

		return locations;
	}

	/**
	 * Renders a single crumb dom
	 */
	render(baseName, fullPath){
		const dom = this.getCrumbDOM(baseName, fullPath);
		this.container.append(dom);
	}

	/**
	 * Rebuilds the breadcrumbs to match the current location
	 */
	refreshLocation(){
		this.container.innerHTML = "";
		const directoryPaths = this.getCurrentDirectoryAsArrayPath();
		for (const pathInfo of directoryPaths){
			this.render(pathInfo.baseName, pathInfo.fullDirectory);
		}
	}
}

export default BreadcrumbHandler;