import Endpoints from "../../Endpoints.js";
import ImageGallery from "../ImageGallery.js";

class GalleryMember{
	static isDraggingAMember = false;
	static currentlyDraggedMember = null;
	static currentlyDraggedGhostDOM = null;
	static container = document.querySelector("#manage-gallery-members-container");

	/** @type {GalleryMember[]} */
	static members = [];

	static clearMembers(){
		for (const member of GalleryMember.members){
			member.dom.remove();
		}

		// Clear memory
		this.members = [];
	}

	static async saveDragOrder(){
		const payload = [];
		const allMemberDOMs = Array.from(GalleryMember.container.childNodes);
		const galleryID = ImageGallery.manageGalleryModal.currentGalleryID;
		for (const member of GalleryMember.members){
			payload.push({
				id:member.id,
				position:allMemberDOMs.indexOf(member.dom)
			});
		}

		const headers = new Headers();
		headers.set("content-type", "application/json");
		const response = await fetch(`${Endpoints.galleries.updateGalleryMembersPositions}/${galleryID}`, {
			credentials:"same-origin",
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			headers:headers
		});

		let data;
		try{
			/** @type {{error: ?string, status: int}} */
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("Response from server is not valid JSON.");
		}

		if (data.status === 1){
			// Saved
		}else if (data.status === -1){
			alert(data.error);
		}
	}

	constructor(id, fileName, thumbURI, altText){
		this.id = id;
		this.fileName = fileName;
		this.thumbURI = thumbURI;
		this.altText = altText;
		this.dom = this.getDOM();

		// Add it to the DOM
		GalleryMember.container.append(this.dom);
		GalleryMember.members.push(this);
	}

	removeDOM(){
		for (const [index, member] of GalleryMember.members.entries()){
			if (member.id === this.id){
				GalleryMember.members.splice(index, 1);
				break;
			}
		}

		this.dom.remove();
	}

	/**
	 * @return {HTMLDivElement}
	 */
	getDOM(){
		const alt = (this.altText !== "" ? this.altText : "[Auto generated]");
		const template = document.createElement("div");
		template.classList.add("gallery-member-item");
		template.innerHTML = `
			<div class="drag-handle-container">
				<button type="button">
					<i class="bi bi-grip-horizontal"></i>
				</button>
			</div>
			<div class="thumb">
				<img src="${this.thumbURI}" alt="">
			</div>
			<div class="name">
				<strong>File: </strong><span>${this.fileName}</span><br>
				<strong>Alt: </strong><span class="gallery-member-alt-text">${alt}</span>
				<form class="gallery-member-alt-text-form" style="display:none;">
					<input type="hidden" value="${this.id}" name="id">
					<input type="text" class="form-control form-control-sm" name="alt-text" value="${this.altText}">
				</form>
			</div>
			<div class="control-buttons">
				<button type="button" class="delete-gallery-member-button">
					<i class="bi bi-folder-minus"></i>
					<span>Remove</span>
				</button>
			</div>
		`;

		const removeMemberbutton = template.querySelector(".delete-gallery-member-button");

		template.querySelector(".drag-handle-container button").addEventListener("mousedown", e => {
			this.onDragHandleButtonDown(e);
		});

		template.querySelector(".gallery-member-alt-text").addEventListener("click", e => {
			this.onAltTextSpanClicked();
		});

		template.querySelector(".gallery-member-alt-text-form").addEventListener("submit", e => {
			e.preventDefault();
			this.onAltTextFormSubmitted();
		});

		template.querySelector(`[name="alt-text"]`).addEventListener("blur", e => {
			this.onAltTextFormSubmitted();
		});

		removeMemberbutton.addEventListener("click", () => {
			this.onMemberRemoveButtonClicked();
		});

		return template;
	}

	async onMemberRemoveButtonClicked(){

		this.removeDOM();

		const response = await fetch(`/uplift/gallery-manager/gallery/${ImageGallery.manageGalleryModal.currentGalleryID}/members/${this.id}`, {
		    method:"DELETE",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		let data;
		try{
		    /** @type {{status: int, error: ?string}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		    return;
		}

		if (data.status === 1){

		}else if (data.status === -1){

		}
	}

	onAltTextSpanClicked(){
		const altTextSpan = this.dom.querySelector(".gallery-member-alt-text");
		const altTextForm = this.dom.querySelector(".gallery-member-alt-text-form");

		altTextSpan.style.display = "none";
		altTextForm.style.display = null;

		altTextForm.querySelector(`[name="alt-text"]`).focus();
	}

	async onAltTextFormSubmitted(){
		const galleryID = ImageGallery.manageGalleryModal.currentGalleryID;
		const altTextForm = this.dom.querySelector(".gallery-member-alt-text-form");
		const altTextSpan = this.dom.querySelector(".gallery-member-alt-text");
		const newValue = this.dom.querySelector(`[name="alt-text"]`).value.trim();
		if (newValue.length === 0){
			altTextSpan.textContent = "[Auto generated]";
		}else{
			altTextSpan.textContent = newValue;
		}

		// Revert back to the span view
		altTextSpan.style.display = null;
		altTextForm.style.display = "none";

		// PATCH it
		const fData = new FormData(altTextForm);
		const response = await fetch(`${Endpoints.galleries.updateGalleryMemberAltText}/${galleryID}/members/${this.id}/alt-text`, {
			credentials:"same-origin",
			body:fData,
			method:"PATCH",
			cache:"no-cache",
		});

		let data;
		try{
			/** @type {{error: ?string, status: int}} */
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("Response from server is not valid JSON.");
		}

		if (data.status === 1){
			// Saved
		}else if (data.status === -1){
			alert(data.error);
		}
	}

	/**
	 * @param {MouseEvent} event
	 */
	onDragHandleButtonDown(event){
		const ghostDOM = this.dom.cloneNode(true);
		GalleryMember.isDraggingAMember = true;
		GalleryMember.currentlyDraggedMember = this;
		GalleryMember.currentlyDraggedGhostDOM = ghostDOM;
		ghostDOM.classList.add("dragged-ghost-dom");
		ghostDOM.style.width = `${this.dom.clientWidth}px`;
		ghostDOM.style.height = `${this.dom.clientHeight}px`;
		ghostDOM.style.top = `${event.pageY+5}px`;
		ghostDOM.style.left = `${event.pageX+5}px`;
		this.dom.classList.add("has-ghost-dragger-dom");
		document.body.append(ghostDOM);

		GalleryMember.container.focus();
		window.addEventListener("mousemove", this.onMouseMoveWhileDragging, {
			passive: true
		});
		window.addEventListener("mouseup", this.onMouseUpWhileDragging, {
			passive: true
		});
		window.addEventListener("mousewheel", this.onMouseWheelWhileDragging, {
			passive: true
		});
	}

	/**
	 * When the mouse wheel event happens during a drag
	 * @param {WheelEvent} event
	 */
	onMouseWheelWhileDragging(event){
		GalleryMember.container.scrollBy({
			top:event.deltaY
		});
	}

	/**
	 * @param {MouseEvent} event
	 */
	onMouseMoveWhileDragging(event){
		GalleryMember.currentlyDraggedGhostDOM.style.top = `${event.pageY+5}px`;
		GalleryMember.currentlyDraggedGhostDOM.style.left = `${event.pageX+5}px`;

		// Determine which sibling order the dragged element can change
		// For example, if the current dragged element is the first child, then
		// clearly it can't be dragged upwards
		const distanceFromScreenTopToContainer = GalleryMember.container.getBoundingClientRect().top;
		const howFarUserHasScrolledInContainer = GalleryMember.container.scrollTop;
		const currentlyDraggedMember = GalleryMember.currentlyDraggedMember;
		const prevSibling = currentlyDraggedMember.dom.previousElementSibling;
		const nextSibling = currentlyDraggedMember.dom.nextElementSibling;
		const mousePositionRelativeToContainer = event.clientY - distanceFromScreenTopToContainer;

		if (prevSibling) {
			const prevSiblingPositionRelativeToContainer = prevSibling.offsetTop - howFarUserHasScrolledInContainer;
			if (mousePositionRelativeToContainer < prevSiblingPositionRelativeToContainer) {
				// Swap them
				GalleryMember.container.insertBefore(currentlyDraggedMember.dom, prevSibling);
			}
		}
		if (nextSibling) {
			const nextSiblingPositionRelativeToContainer = nextSibling.offsetTop - howFarUserHasScrolledInContainer;
			if (mousePositionRelativeToContainer > nextSiblingPositionRelativeToContainer) {
				// Move current dragged DOM to after the next sibling
				if (nextSibling.nextElementSibling === null) {
					GalleryMember.container.append(currentlyDraggedMember.dom);
				} else {
					GalleryMember.container.insertBefore(currentlyDraggedMember.dom, nextSibling.nextElementSibling);
				}
			}
		}

		/**
		 * Deprecated on 3/29/2022 below code in favor of a mousewheel event instead
		 * TODO May uncomment for touch screen devices in the far future
		 */
		// If the mouse is near the very top of the scrollable container, scroll it upwards
		// const maxScrollY = GalleryMember.container.scrollHeight - GalleryMember.container.clientHeight;
		// if (
		// 	mousePositionRelativeToContainer < 20 &&
		// 	GalleryMember.container.scrollTop > 0
		// ){
		// 	GalleryMember.container.scrollBy({
		// 		top:-1
		// 	});
		// }else if (
		// 	mousePositionRelativeToContainer > (GalleryMember.container.clientHeight - 20) &&
		// 	GalleryMember.container.scrollTop < maxScrollY
		// ){
		// 	GalleryMember.container.scrollBy({
		// 		top:1
		// 	});
		// }
	}

	/**
	 * @param {MouseEvent} event
	 */
	onMouseUpWhileDragging(event){
		GalleryMember.currentlyDraggedMember.cancelDrag();
	}

	/**
	 * Drag is over, save the current order
	 */
	cancelDrag(){
		window.removeEventListener("mousemove", this.onMouseMoveWhileDragging);
		window.removeEventListener("mouseup", this.onMouseUpWhileDragging);
		window.removeEventListener("mousewheel", this.onMouseWheelWhileDragging);
		GalleryMember.isDraggingAMember = false;
		GalleryMember.currentlyDraggedGhostDOM.remove();
		GalleryMember.currentlyDraggedMember.dom.classList.remove("has-ghost-dragger-dom");
		GalleryMember.currentlyDraggedMember = null;
		GalleryMember.saveDragOrder();
	}
}

export default GalleryMember;