import SearchForm from "../forms/SearchForm.js";

class Result{

	static resultsContainer = document.querySelector("#search-results-container");

	/** @type {Result[]} */
	static resultsCache = [];
	static localCounter = 0;

	filePath = null;
	fileName = null;
	findResults = [];
	pageID = null;
	pageName = null;
	type = "";
	dom = null;
	isProcessing = false;

	static clearAll(){
		for (const result of Result.resultsCache){
			result.dom.remove();
		}

		Result.resultsCache = [];
	}

	/**
	 * @param {{filePath: string|null, findResults:Object[], pageID: int|null, type:string, fileName: string|null, pageName: string|null}} findResult
	 */
	constructor(findResult){

		// Local ID to mark an individual Result item to be deleted or identified
		this.localID = Result.localCounter++;
		this.originalResult = findResult;
		this.filePath = findResult.filePath;
		this.fileName = findResult.fileName;
		this.findResults = findResult.findResults;
		this.pageID = findResult.pageID;
		this.pageName = findResult.pageName;
		this.type = findResult.type;

		if (this.type === "PAGE") {
			this.dom = this.getPageResultDOM();
		}else{
			this.dom = this.getFileResultDOM();
		}

		Result.resultsContainer.append(this.dom);
		Result.resultsCache.push(this);
	}

	remove(){
		this.dom.remove();
		for (const resultIndex in Result.resultsCache){
			const iteratedResult = Result.resultsCache[resultIndex];
			if (iteratedResult.localID === this.localID){
				Result.resultsCache.splice(parseInt(resultIndex), 1);
			}
		}
	}

	removeOnlyHead(){
		this.dom.querySelector(".replace-specific-page-section-buttons-container").innerHTML = "";
		this.dom.querySelector(".page-head-column").remove();
	}

	removeOnlyBody(){
		this.dom.querySelector(".replace-specific-page-section-buttons-container").innerHTML = "";
		this.dom.querySelector(".page-body-column").remove();
	}

	getPageResultDOM(){
		let title = `Page: ${this.pageName} (ID: ${this.pageID})`;

		const headResultItems = [];
		const bodyResultItems = [];
		for (/** @type {{isPageHead: boolean, start: int, end: int, stub: Object}} */ const findResult of this.findResults){
			/** @type {{before: string, beforeSanitized: string, result: string, resultSanitized: string, after: string, afterSanitized: string}} */
			const stub = findResult.stub;
			const resultItem = document.createElement("div");
			const beforeSpan = document.createElement("span");
			const resultSpan = document.createElement("span");
			const afterSpan = document.createElement("span");

			resultSpan.classList.add("find-replace-search-result-result-container");

			// Use innerHTML and not textContent here so the html entities render as their display characters
			beforeSpan.innerHTML = `${stub.beforeSanitized.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
			resultSpan.innerHTML = stub.resultSanitized;
			afterSpan.innerHTML = `${stub.afterSanitized}${stub.afterSanitized.length > 0 ? "... " : ""}`;

			resultItem.append(beforeSpan);
			resultItem.append(resultSpan);
			resultItem.append(afterSpan);

			if (findResult.isPageHead){
				headResultItems.push(resultItem);
			}else{
				bodyResultItems.push(resultItem);
			}
		}

		const template = document.createElement("div");
		template.classList.add("find-and-replace-find-result-item");
		template.innerHTML = `
			<div class="card">
				<div class="card-header">
					<div class="d-flex justify-content-between align-items-center">
						<h4>${title}</h4>
						<a target="_blank" href="/uplift/page-editor/${this.pageID}" class="text-decoration-none">
							<span>Edit</span>
							<i class="bi bi-pencil-square"></i>
						</a>
					</div>
				</div>
				<div class="card-header">
					<div class="d-flex justify-content-between">
						<div class="replace-specific-page-section-buttons-container">
							<button type="button" class="btn btn-sm btn-primary replace-body-occurrences-button">
								<i class="bi bi-arrow-repeat"></i>
								<span>Replace All in Body</span>
							</button>
							<button type="button" class="btn btn-sm btn-primary replace-head-occurrences-button">
								<i class="bi bi-arrow-repeat"></i>
								<span>Replace All in Head</span>
							</button>
						</div>
						<div class="text-end">
							<button type="button" class="btn btn-sm btn-primary replace-all-occurrences-button">
								<i class="bi bi-arrow-repeat"></i>
								<span>Replace All</span>
							</button>
						</div>
					</div>
				</div>
				<div class="card-body">
					<div class="container-fluid">
						<div class="row">
							<div class="col-xl-6 page-body-column">
								<h5>Page Body</h5>
								<div class="body-result-stub-container"></div>
							</div>
							<div class="col-xl-6 page-head-column">
								<h5>Page Head</h5>
								<div class="head-result-stub-container"></div>
							</div>
						</div>
					</div>
				</div>
			</div>
		`;

		const headStubContainer = template.querySelector(".head-result-stub-container");
		const bodyStubContainer = template.querySelector(".body-result-stub-container");

		for (const item of headResultItems){
			headStubContainer.append(item);
		}

		for (const item of bodyResultItems){
			bodyStubContainer.append(item);
		}

		const replaceAllButton = template.querySelector(".replace-all-occurrences-button");
		const replaceAllBodyButton = template.querySelector(".replace-body-occurrences-button");
		const replaceAllHeadButton = template.querySelector(".replace-head-occurrences-button");

		replaceAllButton.addEventListener("click", () => {
			this.replaceAllOccurrenceButtonClicked();
		});

		replaceAllBodyButton.addEventListener("click", () => {
			this.replaceAllBodyOccurrenceButtonClicked();
		});

		replaceAllHeadButton.addEventListener("click", () => {
			this.replaceAllHeadOccurrenceButtonClicked();
		});

		return template;
	}

	getFileResultDOM(){
		let title = `Theme File: ${this.fileName}`;

		const resultItems = [];
		for (/** @type {{isPageHead: boolean, start: int, end: int, stub: Object}} */ const findResult of this.findResults){
			/** @type {{before: string, beforeSanitized: string, result: string, resultSanitized: string, after: string, afterSanitized: string}} */
			const stub = findResult.stub;
			const resultItem = document.createElement("div");
			const beforeSpan = document.createElement("span");
			const resultSpan = document.createElement("span");
			const afterSpan = document.createElement("span");

			resultSpan.classList.add("find-replace-search-result-result-container");

			// Use innerHTML and not textContent here so the html entities render as their display characters
			beforeSpan.innerHTML = `${stub.before.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
			resultSpan.innerHTML = stub.resultSanitized;
			afterSpan.innerHTML = `${stub.afterSanitized}${stub.after.length > 0 ? "... " : ""}`;

			resultItem.append(beforeSpan);
			resultItem.append(resultSpan);
			resultItem.append(afterSpan);

			resultItems.push(resultItem);
		}

		const template = document.createElement("div");
		template.classList.add("find-and-replace-find-result-item");
		template.innerHTML = `
			<div class="card">
				<div class="card-header">
					<h4>${title}</h4>
				</div>
				<div class="card-body">
					<div>
						<div class="result-stub-container"></div>
					</div>
				</div>
				<div class="card-footer">
					<div class="d-flex justify-content-end">
						<div class="text-end">
							<button type="button" class="btn btn-sm btn-primary replace-file-occurrences-button">
								<i class="bi bi-arrow-repeat"></i>
								<span>Replace All</span>
							</button>
						</div>
					</div>
				</div>
			</div>
		`;

		const stubContainer = template.querySelector(".result-stub-container");
		for (const item of resultItems){
			stubContainer.append(item);
		}

		const replaceAllButton = template.querySelector(".replace-file-occurrences-button");
		replaceAllButton.addEventListener("click", () => {
			this.replaceAllOccurrenceButtonClicked();
		});

		return template;
	}

	async replaceAllOccurrenceButtonClicked(){

		if (this.isProcessing){
			return;
		}

		if (SearchForm.searchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find operation. You should re-run the Find operation before trying to make a replacement.");
			return;
		}

		this.isProcessing = true;

		const payload = {};
		payload.findResult = this.originalResult;
		payload.replacement = document.querySelector("#replacement-input").value;

		const response = await fetch(`/uplift/find-and-replace/replace`, {
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin",
			headers:{
				"Content-Type":"application/json"
			}
		});

		let data;
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.isProcessing = false;
			return;
		}

		if (data.status === 1){
			this.remove();
		}else if (data.status === -1){
			alert(data.error);
		}

		this.isProcessing = false;
	}

	async replaceAllBodyOccurrenceButtonClicked(){

		if (this.isProcessing){
			return;
		}

		if (SearchForm.searchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find operation. You should re-run the Find operation before trying to make a replacement.");
			return;
		}

		this.isProcessing = true;

		const payload = {};
		const resultClone = structuredClone(this.originalResult);

		// Get a new array of find result items that are only body replacements
		const newFindResultItems = [];
		const leftOverFindResultItems = [];
		for (const findResultItem of resultClone.findResults){
			if (!findResultItem.isPageHead){
				newFindResultItems.push(findResultItem);
			}else{
				leftOverFindResultItems.push(findResultItem);
			}
		}

		resultClone.findResults = newFindResultItems;

		payload.findResult = resultClone;
		payload.replacement = document.querySelector("#replacement-input").value;

		const response = await fetch(`/uplift/find-and-replace/replace`, {
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin",
			headers:{
				"Content-Type":"application/json"
			}
		});

		let data;
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.isProcessing = false;
			return;
		}

		if (data.status === 1){
			this.originalResult.findResults = leftOverFindResultItems;
			this.removeOnlyBody();
		}else if (data.status === -1){
			alert(data.error);
		}

		this.isProcessing = false;
	}

	async replaceAllHeadOccurrenceButtonClicked(){

		if (this.isProcessing){
			return;
		}

		if (SearchForm.searchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find operation. You should re-run the Find operation before trying to make a replacement.");
			return;
		}

		this.isProcessing = true;

		const payload = {};
		const resultClone = structuredClone(this.originalResult);

		// Get a new array of find result items that are only head replacements
		const newFindResultItems = [];
		const leftOverFindResultItems = [];
		for (const findResultItem of resultClone.findResults){
			if (findResultItem.isPageHead){
				newFindResultItems.push(findResultItem);
			}else{
				leftOverFindResultItems.push(findResultItem);
			}
		}

		resultClone.findResults = newFindResultItems;

		payload.findResult = resultClone;
		payload.replacement = document.querySelector("#replacement-input").value;

		const response = await fetch(`/uplift/find-and-replace/replace`, {
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin",
			headers:{
				"Content-Type":"application/json"
			}
		});

		let data;
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.isProcessing = false;
			return;
		}

		if (data.status === 1){
			this.originalResult.findResults = leftOverFindResultItems;
			this.removeOnlyHead();
		}else if (data.status === -1){
			alert(data.error);
		}

		this.isProcessing = false;
	}
}

export default Result;