import PageEditorStates from "../PageEditorStates.js";
import EditorEndpoints from "../../EditorEndpoints.js";
import ConfirmModal from "../../../utils/ConfirmModal.js";

/**
* Component class for the blog category dropdown and its
* various functions (create, rename, delete)
* and helpers for the select menu belonging to it
*/
class BlogCategorySelectDropdown{
	constructor(){
		if (PageEditorStates.currentPageType === "Blog"){
			this.input = document.querySelector("#blog-categories-selection");
			this.renameButton = document.querySelector("#rename-blog-category-button");
			this.deleteButton = document.querySelector("#delete-blog-category-button");

			this.input.addEventListener("input", e => {
				this.onInput(e);
			});

			this.deleteButton.addEventListener("click", () => {
				this.onDeleteCategoryButtonClicked();
			});

			// Call this upon class init to check for
			// a preloaded blog category
			this.onInput();
		}
	}

	/**
	* When the input event fires for the input
	* of this singleton
	* @param {Event|undefined} e Will be undefined if this function was manually invoked
	*/
	onInput(e){
		const currentValue = this.input.value;
		if (currentValue != 0){
			this.showCategoryOptionButtons();
		}else{
			this.hideCategoryOptionButtons();
		}
	}

	/**
	* When the delete category button is clicked
	*/
	async onDeleteCategoryButtonClicked(){
		const categoryID = this.input.value;
		const currentCategoryName = this.getTextOfSelectedOption();
		const confirmModal = new ConfirmModal("delete-category-modal");
		confirmModal.setTitle("Confirm category deletion");
		confirmModal.setContent(`Are you sure you want to delete the category ${currentCategoryName}? Any blogs that use this category will be affected.`);
		confirmModal.setConfirmButtonText("Confirm Deletion");
		confirmModal.showModal();

		// True if they confirm
		// False if they close the modal in any way
		const confirmation = await confirmModal.actionTaken();

		if (confirmation){
			const fData = new FormData();

			fData.append("category-id", categoryID);

			const response = await fetch(EditorEndpoints.deleteBlogCategory, {
				cache:"no-cache",
				credentials:"same-origin",
				method:"post",
				body:fData
			});

			const data = await response.json();

			// Remove the modal regardless if the action succeeded or not
			confirmModal.hideModal();

			if (data.status == 1){
				this.removeOptionByValue(categoryID);
				this.onInput();
			}
		}
	}

	/**
	* Removes the selected attribute from whichever
	* option has it
	*/
	clearSelectedValue(){
		const options = this.input.querySelectorAll("option");

		for (let option of options){
			if (option.hasAttribute("selected")){
				option.removeAttribute("selected");
			}
		}
	}

	/**
	* Removes an option that has a given value
	* @returns {string}
	*/
	removeOptionByValue(value){
		const options = this.input.querySelectorAll("option");

		for (let option of options){
			if (option.getAttribute("value") == value){
				option.remove();
				break;
			}
		}
	}

	/**
	* Returns the text content (label) of the currently selected
	* option
	* @returns {string}
	*/
	getTextOfSelectedOption(){
		const options = this.input.querySelectorAll("option");

		for (let option of options){
			if (option.hasAttribute("selected") || option.getAttribute("value") == this.input.value){
				return option.textContent;
			}
		}
	}

	/**
	* Adds an option to this <select> element
	* @param {string} value
	* @param {string} label
	* @param {boolean} selected (optional)
	*/
	addOption(value, label, selected){
		const newOption = document.createElement("option");
		newOption.setAttribute("value", value)
		newOption.textContent = label

		if (selected){
			this.clearSelectedValue();
			newOption.setAttribute("selected", "");
		}

		this.input.append(newOption);
	}

	/**
	* Changes the text of an option identified by its value
	* @param {string} value
	* @param {string} newLabel
	*/
	renameOption(value, newLabel){
		const options = this.input.querySelectorAll("option");

		for (let option of options){
			if (option.getAttribute("value") == value){
				option.textContent = newLabel;
			}
		}
	}

	/**
	* Shows the rename and delete buttons
	*/
	showCategoryOptionButtons(){
		this.renameButton.style.display = "inline-block";
		this.deleteButton.style.display = "inline-block";
	}

	/**
	* Hides the rename and delete buttons
	*/
	hideCategoryOptionButtons(){
		this.renameButton.style.display = "none";
		this.deleteButton.style.display = "none";
	}
}

export default (new BlogCategorySelectDropdown);
