import RecentArticlesShortcodeButton from "../shortcode-buttons/RecentArticles.js";
import PageEditorStates from "../PageEditorStates.js";
import RecentArticlesCategoryButton from "../components/RecentArticlesCategoryButton.js";

class RecentArticlesForm{
	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form = document.querySelector("#recent-articles-form");

		/** @property {bool} isSubmitting The current status of submission */
		this.isSubmitting = false;

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){
		const fData = new FormData(this.form);
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		const errorContainer = this.form.querySelector(".submit-error");

		if (this.isSubmitting){
			return;
		}

		// Show the submit button as disabled
		submitButton.classList.add("disabled");
		errorContainer.style.display = "none";

		// Get all project tags to be included
		const includedArticleCategories = [];
		const includeArticlesCategoryButtonComponents = RecentArticlesCategoryButton.rows;

		for (const buttonComponent of includeArticlesCategoryButtonComponents){
			if (buttonComponent.isSelected){
				includedArticleCategories.push(buttonComponent.categoryID);
			}
		}

		// Get the number of articles per page and columns
		const numRecentArticles = parseInt(String(fData.get("num-recent-articles")));
		const numColumns = parseInt(String(fData.get("recent-articles-num-columns")));
		const autofillArticles = fData.get("recent-articles-autofill"); // "on" or null

		if (isNaN(numRecentArticles)){
			this.onFormattedError("Number of articles is in a non-numerical format.");
			return;
		}

		if (isNaN(numColumns)){
			this.onFormattedError("Number of columns is in a non-numerical format.");
			return;
		}

		this.onSuccess(numRecentArticles, numColumns, includedArticleCategories, autofillArticles);
	}

	/**
	* When the form succeeds without error
	*/
	onSuccess(numRecentArticles, numColumns, includedArticleCategories, autofillArticles){
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		this.isSubmitting = false;
		submitButton.classList.remove("disabled");

		// Inject the shortcode with the known parameters
		RecentArticlesShortcodeButton.injectShortcode(numRecentArticles, numColumns, includedArticleCategories, autofillArticles);
	}

	/**
	* A readable error happened on the server in the JSON response.
	* (A status of -1 was read)
	* @param {string} err
	*/
	onFormattedError(err){
		const errorContainer = this.form.querySelector(".submit-error");
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		errorContainer.textContent = err;
		errorContainer.style.display = "block";
	}
}

export default (new RecentArticlesForm);
