import PageEditorStates from "../PageEditorStates.js";
import ContentChangedHandler from "../../ContentChangedHandler.js";
import ConfirmModal from "../../../utils/ConfirmModal.js";
import AceEditorExtension from "../../AceEditorExtension.js";
import BreadcrumbsManager from "../BreadcrumbsManager.js";
import ArticleCategoryManager from "../ArticleCategoryManager.js";
import ProjectTagManager from "../ProjectTagManager.js";
import PageButton from "../components/PageButton.js";
import PageAttribute from "../components/PageAttribute.js";
import {PageContentSectionEditor} from "../components/PageContentSectionEditor.js";

class PageEditForm{
	form = document.querySelector("#page-edit-form");

	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){

		// Do not try and save while a new page is being loaded into the page editor
		if (PageEditorStates.isLoadingPage){
			return;
		}

		// Do not try and save if a save is already processing
		if (PageEditorStates.isSavingPage){
			return;
		}

		const fData = new FormData(this.form);

		ContentChangedHandler.showSaveSpinners();
		PageEditorStates.isSavingPage = true;

		// Push the ace editor content into the packet
		fData.append("page-body", AceEditorExtension.pageEditors.body.ace.getValue());
		fData.append("page-head", AceEditorExtension.pageEditors.head.ace.getValue());
		fData.append("breadcrumbs-json", JSON.stringify(BreadcrumbsManager.breadcrumbs));
		fData.append("page-attribute-values", JSON.stringify(PageAttribute.getCacheForHttpPayload()));

		// Is there a layout section definition currently set for the page layout?
		// If so, we need to make sure we send the PageContentSection objects to be created/saved
		const pageContentSectionsPayload = [];
		if (PageEditorStates.currentPageLayoutSectionDefinition !== null){
			// Ids will be null if they need to be created
			for (const sectionEditor of PageContentSectionEditor.cache){
				pageContentSectionsPayload.push(sectionEditor.pageContentSectionDto);
			}
		}
		fData.append("page-content-sections", JSON.stringify(pageContentSectionsPayload));

		// Add blog-specific payload
		if (PageEditorStates.currentPageType === "Blog"){
			fData.append("article-categories", JSON.stringify(ArticleCategoryManager.selectedArticleCategories));
		}

		// Add project-specific payload
		if (PageEditorStates.currentPageType === "Project"){
			fData.append("project-tag-ids", JSON.stringify(ProjectTagManager.selectedProjectTagIDs));
		}

		// Capture the publication datetime and turn it into a UNIX timestamp
		const publicationDateTimeValue = String(fData.get("page-publication-timestamp"));
		if (publicationDateTimeValue.length > 0) {
			const dateTime = new Date(publicationDateTimeValue);

			// Divide by 1000 because it is stored in milliseconds, but we save it on the server in seconds
			fData.set("page-publication-timestamp", String(dateTime.getTime() / 1000));
		}

		const currentPageID = PageEditorStates.currentPageID;
		const response = await fetch(`/uplift/page-editor/page/${currentPageID}`, {
			body:fData,
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin"
		});

		/** @type {{status: number, error: ?string}} */
		const data = await response.json();
		if (data.status === 1){
			/** @var {{status: number, updatedPageContentSections: {id: number, content: string, pageId: number, sectionName: string}[]|null}} data */
			PageEditorStates.markChangesSaved();
			ContentChangedHandler.hideSaveSpinners();
			ContentChangedHandler.hideAllUnsavedChangesIdentifiers();

			// Update the PageButton page name label to whatever the current page is
			const pageButton = PageButton.getPageButtonByPageID(currentPageID);
			pageButton.dom.querySelector(".page-label-span").textContent = document.querySelector("#page-name").value.trim();

			// Update the View Page button's href
			document.querySelector("#view-page-external-link-button").setAttribute("href", fData.get("page-url").trim());

			// If we had sent PageContentSection objects, then the server will respond with a pageContentSections payload key which will
			// contain those objects back, but with any null Ids now assigned because they were created in the database.

			// We need to match the sectionNames in the return payload to sectionNames we have in our editor and assign
			// ids to them if they were null.
			if (data.updatedPageContentSections !== null){
				if (data.updatedPageContentSections.length > 0) {
					// Update the stored page content sections, if there are any to update
					PageEditorStates.lastLoadedPageContentSections = data.updatedPageContentSections;
					for (const savedPageContentSection of data.updatedPageContentSections) {
						/** @type {PageContentSectionEditor|null} */
						let pageContentEditorWithMatchingSectionName = null;
						for (const component of PageContentSectionEditor.cache) {
							if (component.pageContentSectionDto.sectionName === savedPageContentSection.sectionName) {
								pageContentEditorWithMatchingSectionName = component;
								break;
							}
						}

						if (pageContentEditorWithMatchingSectionName !== null) {
							// Update the pageContentSectionDto
							pageContentEditorWithMatchingSectionName.pageContentSectionDto.id = savedPageContentSection.id;
						}
					}
				}
			}
		}else if (data.status === -1){
			// Make a modal notifying the user of the issue
			const errorMessage = data.error;
			const modal = new ConfirmModal("save-error-modal");
			modal.setTitle("Save error");
			modal.setContent(errorMessage);
			modal.setConfirmButtonText("Okay");
			modal.hideSecondaryButton();
			modal.showModal();
			await modal.actionTaken();
			modal.hideModal();
			modal.modal.remove();
			ContentChangedHandler.hideSaveSpinners();
		}

		PageEditorStates.isSavingPage = false;
	}
}

export default (new PageEditForm)
