import IPPReelShortcodeButton from "../shortcode-buttons/IPPReel.js";
import PageEditorStates from "../PageEditorStates.js";

class IPPReelForm{
	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form = document.querySelector("#ipp-reel-form");

		/** @property {bool} isSubmitting The current status of submission */
		this.isSubmitting = false;

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){
		const fData = new FormData(this.form);
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		const errorContainer = this.form.querySelector(".submit-error");

		if (this.isSubmitting){
			return;
		}

		// Show the submit button as disabled
		submitButton.classList.add("disabled");
		errorContainer.style.display = "none";

		// Get all project tags included and excluded
		const includedTagIDs = [];
		const excludedTagIDs = [];
		const includeTagButtons = this.form.querySelectorAll(".ipp-reel-tag-include-button");
		const excludeTagButtons = this.form.querySelectorAll(".ipp-reel-tag-exclude-button");

		for (let button of includeTagButtons){
			if (button.classList.contains("selected")){
				includedTagIDs.push(parseInt(button.getAttribute("data-tag-id")));
			}
		}

		for (let button of excludeTagButtons){
			if (button.classList.contains("selected")){
				excludedTagIDs.push(parseInt(button.getAttribute("data-tag-id")));
			}
		}

		// Get the number of articles per page and columns
		const numProjectsPerPage = parseInt(String(fData.get("ipp-reel-projects-per-page")));
		const numColumns = parseInt(String(fData.get("ipp-reel-num-columns")));

		if (isNaN(numProjectsPerPage)){
			this.onFormattedError("Number of projects is in a non-numerical format.");
			return;
		}

		if (isNaN(numColumns)){
			this.onFormattedError("Number of columns is in a non-numerical format.");
			return;
		}

		this.onSuccess(numProjectsPerPage, numColumns, includedTagIDs, excludedTagIDs);
	}

	/**
	* When the form succeeds without error
	*/
	onSuccess(numProjectsPerPage, numColumns, includedTagIDs, excludedTagIDs){
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		this.isSubmitting = false;
		submitButton.classList.remove("disabled");

		// Inject the shortcode with the known parameters
		IPPReelShortcodeButton.injectShortcode(numProjectsPerPage, numColumns, includedTagIDs, excludedTagIDs);
	}

	/**
	* A readable error happened on the server in the JSON response.
	* (A status of -1 was read)
	* @param {string} err
	*/
	onFormattedError(err){
		const errorContainer = this.form.querySelector(".submit-error");
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		errorContainer.textContent = err;
		errorContainer.style.display = "block";
	}
}

export default new IPPReelForm();
