import PageEditorStates from "../PageEditorStates.js";
import Endpoints from "../../../Endpoints.js";

class ClonePageForm{
	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form = document.querySelector("#clone-page-form");

		/** @property {bool} isSubmitting The current status of submission */
		this.isSubmitting = false;

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){
		const fData = new FormData(this.form);
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		const errorContainer = this.form.querySelector(".submit-error");

		if (this.isSubmitting){
			return;
		}

		if (PageEditorStates.unsavedChanges){
			if (!confirm("Creating a new page will reload the page, and you have unsaved changes. Are you sure?")){
				return;
			}
		}

		// Show the submit button as disabled
		submitButton.classList.add("disabled");
		errorContainer.style.display = "none";

		const response = await fetch(`/uplift/page-editor/page/${PageEditorStates.currentPageID}/clone`, {
			cache:"no-cache",
			method:"post",
			credentials:"same-origin",
			body:fData
		});

		/** @type {{status: int, clonedPageID: int, error: ?string}} */
		const data = await response.json();

		if (data.status === 1){
			this.onSuccess(data.clonedPageID);
		}else if (data.status === -1){
			this.onFormattedError(data.error);
		}
	}

	/**
	 * When the form receives a success message from the server
	 * (A status of 1 was read)
	 * @param {int} clonedPageID
	 */
	onSuccess(clonedPageID){
		window.location.href = `${Endpoints.pageEditor.editor}/${clonedPageID}`;
	}

	/**
	* A readable error happened on the server in the JSON response.
	* (A status of -1 was read)
	* @param {string} err
	*/
	onFormattedError(err){
		const errorContainer = this.form.querySelector(".submit-error");
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		errorContainer.textContent = err;
		errorContainer.style.display = "block";
	}
}

export default (new ClonePageForm);
