import BlogCategoryRenameModal from "../modals/BlogCategoryRenameModal.js";
import BlogCategoryManageModal from "../modals/BlogCategoryManageModal.js";
import ArticleCategoryRow from "../components/ArticleCategoryRow.js";
import PageEditorStates from "../PageEditorStates.js";

class BlogCategoryRenameForm{
	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form = document.querySelector("#blog-category-rename-form");
		this.oldCategoryName = null;
		this.newCategoryName = null;

		/** @property {bool} isSubmitting The current status of submission */
		this.isSubmitting = false;

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){
		const fData = new FormData(this.form);
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		const errorContainer = this.form.querySelector(".submit-error");

		if (this.isSubmitting){
			return;
		}

		// Show the submit button as disabled
		submitButton.classList.add("disabled");
		errorContainer.style.display = "none";

		const response = await fetch(`/uplift/page-editor/article-category/${fData.get("blog-category-rename-id")}`, {
			cache:"no-cache",
			method:"patch",
			credentials:"same-origin",
			body:fData
		});

		/** @type {{status:int, error: ?string, newCategoryName: ?string}} */
		const data = await response.json();

		if (data.status === 1){
			this.newCategoryName = data.newCategoryName;
			this.onSuccess();
		}else if (data.status === -1){
			this.onFormattedError(data.error);
		}
	}

	/**
	* When the form receives a success (status code 1) from
	* the server
	*/
	onSuccess(){
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		this.form.reset();
		BlogCategoryRenameModal.hideModal();
		BlogCategoryManageModal.showModal();
		ArticleCategoryRow.renameCategory(this.oldCategoryName, this.newCategoryName);
	}

	/**
	* A readable error happened on the server in the JSON response.
	* (A status of -1 was read)
	* @param {string} err
	*/
	onFormattedError(err){
		const errorContainer = this.form.querySelector(".submit-error");
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		errorContainer.textContent = err;
		errorContainer.style.display = "block";
	}
}

export default new BlogCategoryRenameForm();
