import AceEditorExtension from "../../AceEditorExtension.js";
import PageEditorStates from "../PageEditorStates.js";
import ContentChangedHandler from "../../ContentChangedHandler.js";
import DOMHelper from "../../DOMHelper.js";

/**
 * Represents a single instance of an individual section editor component for a page's body content.
 */
export class PageContentSectionEditor{
	/**
	 * A cache of currently rendered PageContentSectionEditor
	 * @type {PageContentSectionEditor[]}
	 */
	static cache = [];

	/**
	 * Clears all the currently rendered PageContentSectionEditor
	 */
	static clearAll(){
		for (const sectionEditor of PageContentSectionEditor.cache){
			sectionEditor.dom.remove();
		}

		PageContentSectionEditor.cache = [];
	}

	/**
	 * The HTML element this component is represented by
	 * @type {HTMLDivElement}
	 */
	dom;
	/**
	 * The page layout section objects that this page content section should reflect. Can't be null
	 * @type {{id: number, pageLayoutSectionDefinitionId: number, sectionName: string, creationTimestamp: number}}
	 */
	pageLayoutSectionDto;
	/**
	 * The object that represents this individual section for this particular page. Can be null
	 * if it has never been saved yet.
	 * @type {{id: number, pageId: number, sectionName: string, content: string} | null}
	 */
	pageContentSectionDto;
	/**
	 * The instance of AceEditorExtension that the editor for this component is using
	 * @type {AceEditorExtension}
	 */
	aceEditorExtension;

	/**
	 * @param {{id: number, creationTimestamp: number, pageLayoutSectionDefinitionId: number, sectionName: string}} pageLayoutSectionDto
	 * @param {{id: number|null, pageId: number, sectionName: string, content: string | null}|null} pageContentSectionDto This can be null
	 */
	constructor(pageLayoutSectionDto, pageContentSectionDto) {
		this.pageLayoutSectionDto = pageLayoutSectionDto;
		this.pageContentSectionDto = pageContentSectionDto;
		this.constructor.cache.push(this);

		// Create a PageContentSection dto if the one we got was null. Just leave the id null for now,
		// as the page saving logic will populate it once the PageContentSection is saved and created
		// on the backend
		if (pageContentSectionDto === null){
			this.pageContentSectionDto = {
				id: null,
				pageId: PageEditorStates.currentPageID,
				sectionName: this.pageLayoutSectionDto.sectionName,
				content: ""
			};
		}
	}

	build(){
		const template = document.createElement("div");
		template.classList.add("page-content-section-editor-component");
		template.innerHTML = `
			<div class="title-container">
				<strong class="me-2">Section: </strong><span>${this.pageLayoutSectionDto.sectionName}</span>
			</div>
			<div class="editor-container">
				<div class="ace-editor-container" style="height:500px"></div>
			</div>
		`;

		this.aceEditorExtension = new AceEditorExtension(template.querySelector(".ace-editor-container"), "html");
		this.aceEditorExtension.loadExtensions();
		this.aceEditorExtension.ace.session.setValue(this.pageContentSectionDto?.content ?? "");

		// Connect a change event to trigger the unsaved changes marker
		this.aceEditorExtension.ace.on("change", () => {
			if (!PageEditorStates.isLoadingPage) {
				ContentChangedHandler.showUnsavedChangeIdentifier(DOMHelper.getAllPageEditorTabs().body);
				PageEditorStates.markAsHavingUnsavedChanges(false);
				this.pageContentSectionDto.content = this.aceEditorExtension.ace.session.getValue();
			}
		});

		// When focused, set it as the current editor in the page states. This
		// tells the system where to insert objects selected from shortcodes I guess
		this.aceEditorExtension.ace.on("focus", () => {
			PageEditorStates.currentEditor = this.aceEditorExtension;
		});

		this.dom = template;
		return this;
	}

	renderInto(container){
		container.append(this.dom);
		this.aceEditorExtension.ace.resize(true);
	}
}