import Endpoints from "../../../Endpoints.js";
import PageEditor from "../PageEditor.js";
import PageEditorStates from "../PageEditorStates.js";
import ContentChangedHandler from "../../ContentChangedHandler.js";

class PageButton{
	pageID;
	pageName;
	dom;
	isActivePage;

	/** @type {PageButton[]} */
	static buttonCache = [];

	/**
	 * Fetches the currently active PageButton, if any
	 * @returns {PageButton | null}
	 */
	static getCurrentActivePageButton(){
		for (const button of PageButton.buttonCache){
			if (button.isActivePage){
				return button;
			}
		}

		return null;
	}

	/**
	 * Fetches a page button by the page ID
	 * @params {int}
	 * @returns {PageButton | null}
	 */
	static getPageButtonByPageID(pageID){
		for (const button of PageButton.buttonCache){
			if (button.pageID === pageID){
				return button;
			}
		}

		return null;
	}

	/**
	 *
	 * @param {number} pageID
	 * @param {string} pageName
	 */
	constructor(pageID, pageName){
		this.isActivePage = false;
		this.pageID = pageID;
		this.pageName = pageName;
		this.dom = this.getDOM();
		PageButton.buttonCache.push(this);
	}

	setAsActive(){
		this.isActivePage = true;
		this.dom.classList.add("active-page");
	}

	setAsInactive(){
		this.isActivePage = false;
		this.dom.classList.remove("active-page");
	}

	getDOM(){
		const template = document.createElement("button");
		template.classList.add("page-list-page-button");
		template.setAttribute("type", "button");
		template.setAttribute("title", this.pageName);
		template.setAttribute("data-bs-toggle", "tooltip");
		// template.setAttribute("href", `${Endpoints.pageEditor.editor}/${this.pageID}`);

		template.innerHTML = `
			<span class="button-columns">
				<span class="active-button-column">
					<span class="active-button-indicator"><i class="bi bi-arrow-bar-right"></i></span>
				</span>
				<span class="button-label-column">
					<span class="page-label-icon-span"><i class="bi bi-file-break-fill"></i></span>
					<span class="page-label-span">${this.pageName}</span>
				</span>
			</span>
		`;

		template.addEventListener("click", () => {
			this.onClick();
		});

		return template;
	}

	async onClick(){
		// First, determine if a page is currently being loaded in or saved
		// If so, don't allow a new page to be loaded
		if (!PageEditorStates.isSavingPage && !PageEditorStates.isLoadingPage) {
			const currentlyActiveButton = PageButton.getCurrentActivePageButton();
			if (currentlyActiveButton !== null) {
				if (currentlyActiveButton.pageID === this.pageID) {
					// Same button. Ignore this click
					return;
				}
			}

			// Check if there are unsaved changes
			if (PageEditorStates.unsavedChanges){
				if (!confirm("You have unsaved changes. Are you sure you want to load a new page?")){
					return;
				}else{
					// Remove unsaved changes flag - they confirmed they don't care
					PageEditorStates.unsavedChanges = false;
					ContentChangedHandler.hideAllUnsavedChangesIdentifiers();
				}
			}

			if (currentlyActiveButton !== null) {
				// Deactivate currently active button
				currentlyActiveButton.setAsInactive();
			}
			this.setAsActive();
			await PageEditor.loadPage(this.pageID);
		}
	}
}

export default PageButton;