import BlogCategoryRenameModal from "../modals/BlogCategoryRenameModal.js";
import BlogCategoryManageModel from "../modals/BlogCategoryManageModal.js";
import BlogCategoryRenameForm from "../forms/BlogCategoryRenameForm.js";
import ConfirmModal from "../../../utils/ConfirmModal.js";
import Endpoints from "../../../Endpoints.js";
import PageEditorStates from "../PageEditorStates.js";
import ArticleCategoryManager from "../ArticleCategoryManager.js";

class ArticleCategoryRow{

	/** @type {ArticleCategoryRow[]} */
	static rows = [];

	/**
	 * Changes the VISUALS ONLY of all rows to be shown as deselected.
	 */
	static showAllAsDeselected(){
		for (const row of ArticleCategoryRow.rows){
			row.showAsDeselected();
		}
	}

	/**
	 * Renames a row to a new name
	 * @param {string} oldName
	 * @param {string} newName
	 */
	static renameCategory(oldName, newName){
		for (const articleCategoryRow of ArticleCategoryRow.rows){
			if (articleCategoryRow.articleCategoryName === oldName){
				articleCategoryRow.articleCategoryNameElement.textContent = newName;
				articleCategoryRow.articleCategoryName = newName;
			}
		}
	}

	/**
	 * Fetches an ArticleCategoryRow by the categoryID
	 * @param {int} categoryID
	 * @returns {ArticleCategoryRow | null}
	 */
	static getByArticleCategoryID(categoryID){
		for (const articleCategoryRow of ArticleCategoryRow.rows){
			if (articleCategoryRow.articleCategoryID === categoryID){
				return articleCategoryRow;
			}
		}

		return null;
	}

	/**
	 * @param {number} articleCategoryID
	 * @param {string} articleCategoryName
	 * @param {boolean} selected
	 */
	constructor(articleCategoryID, articleCategoryName, selected){
		// Add self to the static collection
		ArticleCategoryRow.rows.push(this);
		this.articleCategoryID = articleCategoryID;
		this.articleCategoryName = articleCategoryName;
		this.isSelected = selected;
		this.row = this.getDOM();
		this.articleCategoryNameElement = this.row.querySelector(".article-category-name");
		this.categoryNotSelected = this.row.querySelector(".category-not-selected");
		this.categorySelected = this.row.querySelector(".category-selected");

		this.renameButton = this.row.querySelector(".rename-category-button");
		this.deleteButton = this.row.querySelector(".delete-category-button");

		this.row.addEventListener("click", e => {
			// Do not fire on columns that have buttons in them
			if (e.target.closest("column").querySelector("button") === null){
				this.onRowClicked();
			}
		});

		this.renameButton.addEventListener("click", e => {
			this.onRenameButtonClicked();
		});

		this.deleteButton.addEventListener("click", e => {
			this.onDeleteButtonClicked();
		});

		// Init pre-selection
		if (this.isSelected){
			this.showAsSelected();
		}
	}

	showAsSelected(){
		this.row.classList.add("selected");
		this.categoryNotSelected.style.display = "none";
		this.categorySelected.style.display = "inline";
	}

	showAsDeselected(){
		this.row.classList.remove("selected");
		this.categoryNotSelected.style.display = "inline";
		this.categorySelected.style.display = "none";
	}

	onRowClicked(){
		if (this.categoryNotSelected.style.display === "none"){
			// Deselect
			this.showAsDeselected();
			ArticleCategoryManager.removeCategoryFromSelected(this.articleCategoryID);
			// fetch(`${Endpoints.pageEditor.saveBlogCategoryToPage}/${this.articleCategoryID}/page/${PageEditorStates.currentPageID}`, {
			// 	credentials:"same-origin",
			// 	method:"delete",
			// 	cache:"no-cache"
			// });
		}else{
			// Select
			this.showAsSelected();
			ArticleCategoryManager.addCategoryToSelected(this.articleCategoryID);
			// fetch(`${Endpoints.pageEditor.saveBlogCategoryToPage}/${this.articleCategoryID}/page/${PageEditorStates.currentPageID}`, {
			// 	credentials:"same-origin",
			// 	method:"patch",
			// 	cache:"no-cache"
			// });
		}
	}

	async onDeleteButtonClicked(){
		const confirmModal = new ConfirmModal("confirm-delete-category-modal");
		confirmModal.setConfirmButtonHTML(`
			<i class="bi bi-trash-fill"></i>
			<span>Delete</span>
		`);
		confirmModal.setTitle("Confirm category deletion");
		confirmModal.setContent("Deleting a category will also remove its association with any blogs that currently have it assigned.");

		BlogCategoryManageModel.hideModal();
		confirmModal.showModal();
		const wantsToDelete = await confirmModal.actionTaken();

		if (wantsToDelete){
			this.row.remove();
			fetch(`/uplift/page-editor/article-category/${this.articleCategoryID}`, {
				credentials: "same-origin",
				cache: "no-cache",
				method: "delete"
			});
			ArticleCategoryManager.removeCategoryFromSelected(this.articleCategoryID);
		}

		confirmModal.cleanup();
		BlogCategoryManageModel.showModal();
	}

	onRenameButtonClicked(){
		BlogCategoryRenameForm.oldCategoryName = this.articleCategoryName;
		BlogCategoryRenameModal.categoryIDInput.value = String(this.articleCategoryID);
		BlogCategoryRenameModal.renameInput.value = this.articleCategoryName;
		BlogCategoryManageModel.hideModal();
		BlogCategoryRenameModal.showModal();
	}

	/**
	 * @returns {HTMLElement}
	 */
	getDOM(){
		const template = document.createElement("article-category-row");
		template.setAttribute("row-id", String(this.articleCategoryID));
		template.innerHTML = `
			<column>
				<span class="category-not-selected"><i class="bi bi-dash-lg"></i></span>
				<span style="display:none;" class="category-selected text-success"><i class="bi bi-check-lg"></i></span>
			</column>
			<column class="article-name selector-column">
				<span class="article-category-name">${this.articleCategoryName}</span>
			</column>
			<column>
				<div class="btn-group">
					<button type="button" class="rename-category-button btn btn-sm btn-primary">
						<i class="bi bi-input-cursor-text"></i>
						<span>Rename</span>
					</button>
					<button type="button" class="delete-category-button btn btn-sm btn-danger">
						<i class="me-0 bi bi-x-square-fill"></i>
					</button>
				</div>
			</column>
		`;

		document.querySelector("#manage-article-category-rows").append(template);

		return template;
	}
}

export default ArticleCategoryRow;