/**
* Singleton that stores states about the page editor
* that other files/classes can read from
* to get information about the current states of the
* page editor.
*/

class PageEditorStates{
	/**
	 * The current page loaded into the editor
	 * @type {{id:int, pageName: string, pageType: string, pageRoute: string, pageBody: string, pageHead: string, pageRouteIsRegex: int, pageLayout: string, creationTime: int, excludeFromSitemap: int, excludeSchemaInjection: int, publicationStatus: int, publicationTimestamp: int, isDeleted: int, attributes: Object[]} | null}
	 */
	currentPage = null;
	/**
	 * The current page layout section definition loaded into the editor from the current page's selected layout,
	 * if there is one.
	 * @type {{id: number, creationTimestamp: number, layoutFileName: string, pageLayoutSections: Object[]} | null}
	 */
	currentPageLayoutSectionDefinition = null;
	/**
	 * These are the last loaded page content sections. Note that this property is _not_ updated unless a page is
	 * loaded or a page is saved. Editing the content sections does not reflect here until saving.
	 * @type {{id: number, content: string, pageId: number, sectionName: string}[]}
	 */
	lastLoadedPageContentSections = null;

	constructor(){

		/** @type {string} Type of view mode. Either STANDARD or REVISION */
		this.viewType = "";

		if (document.querySelector("#page-editor-viewing-type") !== null) {
			this.viewType = document.querySelector("#page-editor-viewing-type").value;
		}

		/** @type {boolean} Whether or not a page is currently being loaded into the editor */
		this.isLoadingPage = false;

		/** @type {boolean} Whether or not the current page is being saved */
		this.isSavingPage = false;

		/** @property {number|null} */
		this.currentPageID = null;

		/** @property {string|null} currentPageType */
		this.currentPageType = null;

		/** @property {bool} */
		this.unsavedChanges = false;

		/** @property {AceEditorExtension} currentEditor The current instance of the active editor */
		this.currentEditor = null;

		/** @property {?string} */
		this.currentEditorWindowName = "body";

		/** @property {object} */
		this.currentLinkContextMenu = null;
	}

	/**
	 * @param {boolean} forceState Will ignore any if conditions that would otherwise prevent this from marking unsaved changes.
	 */
	markAsHavingUnsavedChanges(forceState){
		// Do not allow this method to mark changes as unsaved if loading in new data
		if (this.isLoadingPage && !forceState){
			return;
		}

		// If this is a revision, no changes are possible anyway
		if (this.viewType === "REVISION"){
			return;
		}

		this.unsavedChanges = true;
	}

	markChangesSaved(){
		this.unsavedChanges = false;
	}
}

export default (new PageEditorStates);
