import BreadcrumbsManager from "./BreadcrumbsManager.js";
import ArticleCategoryManager from "./ArticleCategoryManager.js";
import ArticleCategoryRow from "./components/ArticleCategoryRow.js";
import ProjectTagManager from "./ProjectTagManager.js";
import ProjectTagRow from "./components/ProjectTagRow.js";
import AceEditorExtension from "../AceEditorExtension.js";
import Breadcrumb from "./components/Breadcrumb.js";
import PageEditor from "./PageEditor.js";
import PageEditorStates from "./PageEditorStates.js";
import ContentChangedHandler from "../ContentChangedHandler.js";
import DOMHelper from "../DOMHelper.js";
import PageCity from "./shortcode-buttons/PageCity.js";
import PageState from "./shortcode-buttons/PageState.js";
import PageAttribute from "./components/PageAttribute.js";
import {PageContentSectionEditor} from "./components/PageContentSectionEditor.js";

/**
 * Helper class that will take page data and load it into value fields
 */
class PageContentLoader{
	viewPageButton = document.querySelector("#view-page-external-link-button");
	pageNameInput = document.querySelector("#page-name");
	pageLayoutInput = document.querySelector("#page-layout");
	pageURLInput = document.querySelector("#page-url");
	pageIdDisplay = document.querySelector("#page-id");
	excludeFromSitemapCheckbox = document.querySelector("#exclude-from-sitemap");
	excludeFromSchemaInjection = document.querySelector("#exclude-json-schema");
	pagePublicationInput = document.querySelector("#page-publication");
	pagePublicationDateTimeInput = document.querySelector("#page-publication-timestamp");
	pagePublicationDateTimeContainer = document.querySelector("#page-publication-datetime-container");
	featuredImageContainer = document.querySelector("#page-data-featured-image");
	featuredImage = document.querySelector("#featured-image");
	featuredImageInput = document.querySelector("#featured-image-input");
	featuredImageThumbInput = document.querySelector("#featured-image-thumb-input");
	projectDataContainer = document.querySelector("#page-data-project");
	projectFeaturedImageContainer = document.querySelector("#page-data-project-featured-image-container");
	blogDataContainer = document.querySelector("#page-data-blog");
	blogFeaturedImageContainer = document.querySelector("#page-data-blog-featured-image-container");
	serviceDataContainer = document.querySelector("#page-data-service");
	serviceFeaturedImageContainer = document.querySelector("#page-data-service-featured-image-container");
	cityDataContainer = document.querySelector("#page-data-city");
	cityFeaturedImageContainer = document.querySelector("#page-data-city-featured-image-container");

	// Project type specific fields
	projectCityNameInput = document.querySelector("#city-name");
	projectStateNameInput = document.querySelector("#state-name");
	projectStateNameShorthandInput = document.querySelector("#state-name-shorthand");
	brandsProductsInput = document.querySelector("#brands-products");
	customerTestimonialCheckbox = document.querySelector("#customer-testimonial-check");
	projectCustomerReviewSectionContainer = document.querySelector("#ipp-customer-review-container");
	customerReviewFirstName = document.querySelector("#customer-review-first-name");
	customerReviewLastName = document.querySelector("#customer-review-last-name");
	customerReviewTestimonial = document.querySelector("#customer-testimonial-body");

	// City type specific fields
	cityCityNameInput = document.querySelector("#city-page-city-name");
	cityStateNameInput = document.querySelector("#city-page-state-name");
	cityStateNameShorthandInput = document.querySelector("#city-page-state-name-shorthand");
	cityCityURLInput = document.querySelector("#city-page-city-url");

	// Page attributes
	attributesDataContainer = document.querySelector("#page-data-attributes");

	/**
	 * Loads a page's content into the body editor, head editor, and the page data fields. If a page is sent
	 * with layout sections, then the body editor is not used and we render section components instead.
	 * @param {Object} page
	 * @param {Object[]|null} pageContentSections
	 * @param {Object|null} pageLayoutSectionDefinition
	 */
	loadPage(page, pageContentSections, pageLayoutSectionDefinition){

		// Load the pageBody only if there are no content sections
		if (pageLayoutSectionDefinition !== null){
			this.loadPageContentSections(pageContentSections, pageLayoutSectionDefinition);
		}

		// In both cases, still load the pageBody so any existing pageBody continues to be retained
		AceEditorExtension.pageEditors.body.ace.session.setValue(page.pageBody);

		AceEditorExtension.pageEditors.head.ace.session.setValue(page.pageHead);

		this.viewPageButton.setAttribute("href", page.pageRoute);
		this.pageNameInput.value = page.pageName;

		this.pageURLInput.value = page.pageRoute;
		this.pageIdDisplay.innerHTML = page.id;
		this.excludeFromSitemapCheckbox.checked = page.excludeFromSitemap === 1;
		this.excludeFromSchemaInjection.checked = page.excludeSchemaInjection === 1;

		// Try and infer the page layout from the page type if the page layout is blank
		if (page.pageLayout === "") {
			// Set it to blank anyway to override a previous value from a previous page load
			this.pageLayoutInput.value = "";
			const layoutOptions = this.pageLayoutInput.querySelectorAll("option");
			for (const layoutOption of layoutOptions) {
				if (layoutOption.value === page.pageType) {
					this.pageLayoutInput.value = layoutOption.value;
					PageEditorStates.markAsHavingUnsavedChanges(true);
					ContentChangedHandler.showUnsavedChangeIdentifier(DOMHelper.getAllPageEditorTabs().data);
					break;
				}
			}
		} else {
			this.pageLayoutInput.value = page.pageLayout;
		}

		// For the Convert Page feature, hide the page type this page already is
		for (const option of document.querySelectorAll("#convert-page-new-page-type option")) {
			if (option.value === page.pageType) {
				option.style.display = "none";
			} else {
				option.style.display = null;
			}
		}

		// Load the publication status and time - if not a draft
		// 0 = Draft, 1 = Published
		if (page.publicationStatus === 0) {
			// Draft
			this.pagePublicationInput.value = page.publicationStatus;
			this.pagePublicationDateTimeInput.value = '';
			this.pagePublicationDateTimeContainer.style.display = "none";
		} else if (page.publicationStatus === 1) {
			// Published
			this.pagePublicationInput.value = page.publicationStatus;
			if (page.publicationTimestamp > 0) {
				// Multiply by 1000 because it is saved in seconds, but Date takes milliseconds
				const publicationDate = new Date(page.publicationTimestamp * 1000);

				// Set it to local time, as publicationTimestamp is in UNIX time
				publicationDate.setMinutes(publicationDate.getMinutes() - publicationDate.getTimezoneOffset());
				this.pagePublicationDateTimeInput.value = publicationDate.toISOString().slice(0, 16);
			} else {
				this.pagePublicationDateTimeInput.value = '';
			}

			this.pagePublicationDateTimeContainer.style.display = null;
		}
	}

	/**
	 * Builds and renders page content section editors for each pageLayoutSection in pageLayoutSectionDefinition.
	 * @param {{id: number, pageId: number, sectionName: string, content: string | null}[]|null} pageContentSections
	 * @param {{id: number, creationTimestamp: number, layoutFileName: string, pageLayoutSections: Object[]}|null} pageLayoutSectionDefinition
	 */
	loadPageContentSections(pageContentSections, pageLayoutSectionDefinition){
		/** @type {{id: number, creationTimestamp: number, pageLayoutSectionDefinitionId: number, sectionName: string}[]} */
		const pageLayoutSections = pageLayoutSectionDefinition.pageLayoutSections;
		for (const layoutSection of pageLayoutSections){

			// Get the pageContentSection on this page that matches the pageLayoutSection name, if any
			const pageContentSectionMatchingSectionName = pageContentSections
				.filter(pageContentSection => pageContentSection.sectionName.toLowerCase() === layoutSection.sectionName.toLowerCase());

			const component = new PageContentSectionEditor(layoutSection, pageContentSectionMatchingSectionName[0] ?? null);

			component.build()
				.renderInto(document.querySelector("#page-sections-content-container"));
		}
	}

	loadPageBreadcrumbs(breadcrumbs){
		// Clear existing breadcrumbs
		BreadcrumbsManager.clearBreadcrumbs();

		// Load breadcrumbs
		// Should be ordered correctly from the backend server already
		for (/** @type {{position: int, uri: string, label: string}} */ const breadcrumbData of breadcrumbs){
			const newBreadcrumb = new Breadcrumb(breadcrumbData.label, breadcrumbData.uri);
			BreadcrumbsManager.breadcrumbs.push(newBreadcrumb);
			Breadcrumb.crumbsContainer.append(newBreadcrumb.dom);
		}
	}

	loadPageData(page, pageData){
		// Clear current article selections - if any
		ArticleCategoryManager.clearSelectedCategories();
		ArticleCategoryRow.showAllAsDeselected();

		// Clear current project tag selections - if any
		ProjectTagManager.clearSelectedTagIDs();
		ProjectTagRow.showAllAsDeselected();

		// Show the appropriate data sections
		PageEditor.hideDataSections();

		if (page.pageType === "General"){
			// Hide the Page City and Page State shortcode buttons
			if (PageCity.button) {
				PageCity.button.style.display = "none";
				PageState.button.style.display = "none";
			}
		}else if (page.pageType === "Service"){
			// Hide the Page City and Page State shortcode buttons
			if (PageCity.button) {
				PageCity.button.style.display = "none";
				PageState.button.style.display = "none";
			}

			// Show the featured image and move it to the service featured image container

			// Featured image and service section
			this.featuredImageContainer.style.display = "block";
			this.serviceDataContainer.style.display = "block";
			this.serviceFeaturedImageContainer.append(this.featuredImageContainer);

			// Load the featured image, if set
			if ("FEATURED_IMAGE" in pageData && "FEATURED_IMAGE_THUMB" in pageData){
				this.featuredImage.setAttribute("src", pageData.FEATURED_IMAGE_THUMB);
				this.featuredImageInput.value = pageData.FEATURED_IMAGE;
				this.featuredImageThumbInput.value = pageData.FEATURED_IMAGE_THUMB;
			}else{
				// Clear them
				this.featuredImageContainer.setAttribute("src", null);
				this.featuredImageInput.value = "";
				this.featuredImageThumbInput.value = "";
			}

		}else if (page.pageType === "Blog"){
			// Hide the Page City and Page State shortcode buttons
			if (PageCity.button) {
				PageCity.button.style.display = "none";
				PageState.button.style.display = "none";
			}

			// Show the featured image and move it to the blog featured image container

			// Featured image and blog section, in that order sir
			this.featuredImageContainer.style.display = "block";
			this.blogDataContainer.style.display = "block";
			this.blogFeaturedImageContainer.append(this.featuredImageContainer);

			// Load the featured image, if set
			if ("FEATURED_IMAGE" in pageData && "FEATURED_IMAGE_THUMB" in pageData){
				this.featuredImage.setAttribute("src", pageData.FEATURED_IMAGE_THUMB);
				this.featuredImageInput.value = pageData.FEATURED_IMAGE;
				this.featuredImageThumbInput.value = pageData.FEATURED_IMAGE_THUMB;
			}else{
				// Clear them
				this.featuredImageContainer.setAttribute("src", null);
				this.featuredImageInput.value = "";
				this.featuredImageThumbInput.value = "";
			}

			// Load article categories
			if ("ARTICLE_CATEGORY" in pageData){
				let articleCategoriesToLoad = [];
				if (Array.isArray(pageData.ARTICLE_CATEGORY)){
					articleCategoriesToLoad = pageData.ARTICLE_CATEGORY;
				}else{
					articleCategoriesToLoad.push(pageData.ARTICLE_CATEGORY);
				}

				for (const categoryID of articleCategoriesToLoad){
					ArticleCategoryManager.addCategoryToSelected(parseInt(categoryID));
					const row = ArticleCategoryRow.getByArticleCategoryID(parseInt(categoryID));
					if (row !== null){
						// Show as selected
						row.showAsSelected();
					}
				}
			}
		}else if (page.pageType === "Project"){
			// Show the Page City and Page State shortcode buttons
			if (PageCity.button) {
				PageCity.button.style.display = null;
				PageState.button.style.display = null;
			}

			// Show the featured image and move it to the project featured image container

			// Featured image and project section
			this.featuredImageContainer.style.display = "block";
			this.projectDataContainer.style.display = "block";
			this.projectFeaturedImageContainer.append(this.featuredImageContainer);

			if ("CITY_NAME" in pageData){
				this.projectCityNameInput.value = pageData.CITY_NAME;
			}else{
				this.projectCityNameInput.value = "";
			}

			if ("STATE_NAME" in pageData){
				this.projectStateNameInput.value = pageData.STATE_NAME;
			}else{
				this.projectStateNameInput.value = "";
			}

			if ("STATE_NAME_SHORTHAND" in pageData){
				this.projectStateNameShorthandInput.value = pageData.STATE_NAME_SHORTHAND;
			}else{
				this.projectStateNameShorthandInput.value = "";
			}

			if ("PROJECT_BRANDS_PRODUCTS" in pageData){
				this.brandsProductsInput.value = pageData.PROJECT_BRANDS_PRODUCTS;
			}else{
				this.brandsProductsInput.value = "";
			}

			// Load the featured image, if set
			if ("FEATURED_IMAGE" in pageData && "FEATURED_IMAGE_THUMB" in pageData){
				this.featuredImage.setAttribute("src", pageData.FEATURED_IMAGE_THUMB);
				this.featuredImageInput.value = pageData.FEATURED_IMAGE;
				this.featuredImageThumbInput.value = pageData.FEATURED_IMAGE_THUMB;
			}else{
				// Clear them
				this.featuredImage.setAttribute("src", null);
				this.featuredImageInput.value = "";
				this.featuredImageThumbInput.value = "";
			}

			if ("CUSTOMER_DID_LEAVE_REVIEW" in pageData){
				this.customerTestimonialCheckbox.checked = pageData.CUSTOMER_DID_LEAVE_REVIEW === "1";
				if (pageData.CUSTOMER_DID_LEAVE_REVIEW === "1"){
					this.projectCustomerReviewSectionContainer.style.display = null;
				}else{
					// Disable the IPP customer review container
					// Also clear the customer review fields in case a previously loaded page
					// had those fields populated
					this.projectCustomerReviewSectionContainer.style.display = "none";
					this.customerReviewFirstName.value = "";
					this.customerReviewLastName.value = "";
					this.customerReviewTestimonial.value = "";
				}
			}else{
				this.customerTestimonialCheckbox.checked = false;
				this.projectCustomerReviewSectionContainer.style.display = "none";
				// Because this page has loaded with no "customer review" - clear the fields if a previous
				// page was loaded and had those values populated
				this.customerReviewFirstName.value = "";
				this.customerReviewLastName.value = "";
				this.customerReviewTestimonial.value = "";
			}

			if ("CUSTOMER_REVIEW_FIRST_NAME" in pageData){
				this.customerReviewFirstName.value = pageData.CUSTOMER_REVIEW_FIRST_NAME;
			}else{
				this.customerReviewFirstName.value = "";
			}

			if ("CUSTOMER_REVIEW_LAST_NAME" in pageData){
				this.customerReviewLastName.value = pageData.CUSTOMER_REVIEW_LAST_NAME;
			}else{
				this.customerReviewLastName.value = "";
			}

			if ("CUSTOMER_REVIEW_TESTIMONIAL" in pageData){
				this.customerReviewTestimonial.value = pageData.CUSTOMER_REVIEW_TESTIMONIAL;
			}else{
				this.customerReviewTestimonial.value = "";
			}

			if ("PROJECT_TAG" in pageData){
				let projectTagIDsToLoad = [];
				if (Array.isArray(pageData.PROJECT_TAG)){
					projectTagIDsToLoad = pageData.PROJECT_TAG;
				}else{
					projectTagIDsToLoad.push(pageData.PROJECT_TAG);
				}

				for (const tagID of projectTagIDsToLoad){
					ProjectTagManager.addTagIDToSelected(parseInt(tagID));
					const row = ProjectTagRow.getProjectTagByTagID(parseInt(tagID));
					if (row !== null){
						// Show as selected
						row.showAsSelected();
					}
				}
			}
		}else if (page.pageType === "City"){
			// Show the Page City and Page State shortcode buttons
			if (PageCity.button) {
				PageCity.button.style.display = null;
				PageState.button.style.display = null;
			}

			// Show the featured image and move it to the blog featured image container

			// Featured image and blog section, in that order sir
			this.featuredImageContainer.style.display = "block";
			this.cityDataContainer.style.display = "block";
			this.cityFeaturedImageContainer.append(this.featuredImageContainer);

			// Load the featured image, if set
			if ("FEATURED_IMAGE" in pageData && "FEATURED_IMAGE_THUMB" in pageData){
				this.featuredImage.setAttribute("src", pageData.FEATURED_IMAGE_THUMB);
				this.featuredImageInput.value = pageData.FEATURED_IMAGE;
				this.featuredImageThumbInput.value = pageData.FEATURED_IMAGE_THUMB;
			}else{
				// Clear them
				this.featuredImageContainer.setAttribute("src", null);
				this.featuredImageInput.value = "";
				this.featuredImageThumbInput.value = "";
			}

			// City section
			this.cityDataContainer.style.display = "block";

			// Load city page data
			if ("CITY_NAME" in pageData){
				this.cityCityNameInput.value = pageData.CITY_NAME;
			}else{
				this.cityCityNameInput.value = "";
			}

			if ("STATE_NAME" in pageData){
				this.cityStateNameInput.value = pageData.STATE_NAME;
			}else{
				this.cityStateNameInput.value = "";
			}

			if ("STATE_NAME_SHORTHAND" in pageData){
				this.cityStateNameShorthandInput.value = pageData.STATE_NAME_SHORTHAND;
			}else{
				this.cityStateNameShorthandInput.value = "";
			}

			if ("OFFICIAL_CITY_URL" in pageData){
				this.cityCityURLInput.value = pageData.OFFICIAL_CITY_URL;
			}else{
				this.cityCityURLInput.value = "";
			}
		}
	}

	loadPageAttributes(attributes){

		PageAttribute.clearAll();

		const noResultsMessage = document.querySelector("#no-attributes-message-container");

		// In all cases, show the page attributes data section
		this.attributesDataContainer.style.display = null;

		if (attributes.length > 0){
			noResultsMessage.style.display = "none";

			for (/** @type {{id: int, name: string, value: string, pageAttributeID: int, pageID: int}} */ const attribute of attributes){
				const pageAttribute = new PageAttribute(
					attribute.id,
					attribute.name,
					attribute.value,
					attribute.pageAttributeID
				);

				pageAttribute.render();
			}
		}else{
			noResultsMessage.style.display = null;
		}
	}
}

export default new PageContentLoader();