import FileEditor from "../FileEditor.js";

/**
 * This component represents an individual content section (with edit and delete icons) in the Layout sections modal
 * on the file/template editor.
 */
export class PageSectionComponentRow{
	static nextId = 1;

	/**
	 * Internal Id used for giving unique attributes to HTML elements in this component's dom
	 * @type {number}
	 */
	internalId = PageSectionComponentRow.nextId++;
	/**
	 * @type {{id: number, sectionName: string, content: string, creationTimestamp: number, pageLayoutSectionDefinitionId: number, isDefaultContentSection: number}}
	 */
	dto;
	/**
	 * @type {HTMLDivElement}
	 */
	dom;
	/**
	 * @type {HTMLDivElement}
	 */
	mainContentContainer;
	/**
	 * @type {HTMLDivElement}
	 */
	editFormContainer;
	/**
	 * @type {HTMLInputElement}
	 */
	sectionNameInput;

	/**
	 * @type {boolean}
	 */
	isSaving = false;
	/**
	 * An array of function callbacks that are called when this section finishes saving. Used
	 * in the LayoutSectionsModal.js file to listen for when a section is saved so things like
	 * removing default content badges from other sections can be performed.
	 * @type {(() => void)[]}
	 */
	onFinishedSavingCallbacks = [];

	/**
	 *
	 * @param {{id: number, sectionName: string, content: string, creationTimestamp: number, pageLayoutSectionDefinitionId: number}} dto
	 */
	constructor(dto) {
		this.dto = dto;
	}

	build(){
		const template = document.createElement("div");
		template.classList.add("page-section-row-component");
		template.innerHTML = `
			<div class="edit-form-container" style="display: none;">
				<form id="edit-section-form-${this.internalId}" class="edit-section-form">
					<div class="form-check form-switch">
						<input class="form-check-input default-content-check-input" type="checkbox" role="switch" id="edit-section-default-content-switch-${this.internalId}">
						<label class="form-check-label" for="edit-section-default-content-switch-${this.internalId}">Default content</label>
					</div>
					<div class="input-group input-group-sm">
						<label for="edit-section-name-input-${this.internalId}" class="input-group-text">Section name</label>
						<input id="edit-section-name-input-${this.internalId}" type="text" class="section-name-input form-control form-control-sm">
						<button type="submit" form="edit-section-form-${this.internalId}" class="btn btn-success btn-sm">
							<span>Save</span>
						</button>
						<button type="button" class="cancel-edit-section-button btn btn-text btn-sm">
							<span>Cancel</span>
						</button>
					</div>
				</form>
			</div>
			<div class="main-section">
				<div class="default-badge-container"></div>
				<div class="section-name-container">${this.dto.sectionName}</div>
				<div class="action-buttons">
					<button type="button" class="edit-section-button">
						<i class="bi bi-pencil-fill"></i>
					</button>
					<button type="button" class="delete-section-button">
						<i class="bi bi-trash-fill"></i>
					</button>
				</div>
			</div>
		`;

		this.editFormContainer = template.querySelector(".edit-form-container");
		this.mainContentContainer = template.querySelector(".main-section");
		this.sectionNameInput = template.querySelector(".section-name-input");

		template.querySelector(".edit-section-button").addEventListener("click", () => {
			this.onEditButtonClicked();
		});

		template.querySelector(".cancel-edit-section-button").addEventListener("click", () => {
			this.onCancelEditButtonClicked();
		});

		template.querySelector(".delete-section-button").addEventListener("click", () => {
			this.onDeleteSectionButtonClicked();
		});

		template.querySelector(".edit-section-form").addEventListener("submit", e => {
			e.preventDefault();
			this.onEditFormSubmit();
		});

		if (this.dto.isDefaultContentSection === 1){
			this.editFormContainer.querySelector(".default-content-check-input").checked = true;
			this.renderDefaultContentBadge();
		}

		this.dom = template;
		return this;
	}

	/**
	 * Returns true if this component represents a layout section that is marked as the default content section.
	 */
	isDefaultContentSection(){
		return this.editFormContainer.querySelector(".default-content-check-input").checked;
	}

	/**
	 * Renders the icon that represents this layout section as being the default content section.
	 */
	renderDefaultContentBadge(){
		const defaultContentBadge = document.createElement("div");
		defaultContentBadge.innerHTML = `
				<i class="bi bi-body-text"></i>
			`;

		this.mainContentContainer.querySelector(".default-badge-container").append(defaultContentBadge);
	}

	/**
	 * Used for when this section is updated (or another section is updated) to remove
	 * the default content badge from this component.
	 */
	removeDefaultContentBadge(){
		this.mainContentContainer.querySelector(".default-badge-container").innerHTML = ``;
	}

	/**
	 * Renders this component's dom into the container
	 */
	renderInto(container){
		container.append(this.dom);
		return this;
	}

	onCancelEditButtonClicked(){
		this.mainContentContainer.style.display = null;
		this.editFormContainer.style.display = "none";
	}

	onEditButtonClicked(){
		this.sectionNameInput.value = this.dto.sectionName;
		this.mainContentContainer.style.display = "none";
		this.editFormContainer.style.display = null;
	}

	/**
	 * Adds a function listener to be called when this component finishes a save operation.
	 */
	onFinishedSaving(callback){
		this.onFinishedSavingCallbacks.push(callback);
	}

	async onEditFormSubmit(){
		if (this.isSaving){
			return;
		}

		this.isSaving = true;
		this.sectionNameInput.setAttribute("disabled", "");
		this.sectionNameInput.setAttribute("readonly", "");
		const newSectionName = this.editFormContainer.querySelector(".section-name-input").value;
		const isDefaultSection = this.isDefaultContentSection();

		const fData = new FormData();
		fData.set("newSectionName", newSectionName);
		fData.set("isDefaultContentSection", isDefaultSection ? "1" : "0");

		const response = await fetch(`/uplift/theme-manager/page-layouts/sections-definition/sections/${this.dto.id}`, {
			method: "PATCH",
			body: fData
		});

		if (response.status === 200){
			/** @type {{status: number}} */
			const apiData = await response.json();
			if (apiData.status === 1){
				// Update the local DTO's sectionName and default flag with the one we sent to be saved
				this.dto.sectionName = newSectionName;
				this.dto.isDefaultContentSection = isDefaultSection ? 1 : 0;
				this.dom.querySelector(".section-name-container").textContent = newSectionName;

				if (isDefaultSection){
					this.renderDefaultContentBadge();
				}else{
					this.removeDefaultContentBadge();
				}

				// Click the cancel edit button to hide the edit form
				this.onCancelEditButtonClicked();

				for (const callback of this.onFinishedSavingCallbacks){
					callback();
				}
			}else{
				/** @var {{status: number, error: string}} apiData */
				// Error
				alert(apiData.error);
			}

		}else{
			alert("Internal server error.");
			console.warn(await response.text());
		}

		this.sectionNameInput.removeAttribute("disabled");
		this.sectionNameInput.removeAttribute("readonly");
		this.isSaving = false;
	}

	async onDeleteSectionButtonClicked(){
		const sectionsModal = FileEditor.layoutSectionsModal;
		this.dom.remove();
		const response = await fetch(`/uplift/theme-manager/page-layouts/sections-definition/sections/${this.dto.id}`, {
			method: "DELETE",
		});

		if (sectionsModal.sectionComponentsContainer.childElementCount === 0){
			// Show the no results message
			sectionsModal.noSectionsMessage.style.display = null;

			// There will be no more section definition, since this was the last section deleted
			sectionsModal.currentPageLayoutSectionDefinition = null;
		}
	}
}