import DOMHelper from "../DOMHelper.js";
import FileEditorStates from "./FileEditorStates.js";
import ConfirmModal from "../../utils/ConfirmModal.js";
import SaveFileButton from "./components/SaveFileButton.js";
import {FilePathUtility} from "./FilePathUtility.js";
import FileEditor from "./FileEditor.js";

class EditorFileTab{

	/** @type {HTMLButtonElement} */
	static currentlySelectedButton = null;

	/** @type {{[string]: EditorFileTab}} */
	static registeredButtons = {};

	/**
	 * @returns {null|EditorFileTab}
	 */
	static getActiveTab(){
		const currentlySelectedButton = document.querySelector(".editor-tab.selected");
		if (currentlySelectedButton !== null) {
			const filePath = currentlySelectedButton.getAttribute("file-path");
			return EditorFileTab.getByFilePath(filePath);
		}else{
			return null;
		}
	}

	/**
	 * Gets the EditorFileTab instance that belongs to the provided file path
	 * @param filePath
	 * @returns {EditorFileTab|null}
	 */
	static getByFilePath(filePath){
		if (filePath in EditorFileTab.registeredButtons) {
			return EditorFileTab.registeredButtons[filePath];
		}else{
			return null;
		}
	}

	/**
	 * Fetches the next available edit tab button, that is not the provided one
	 * @param {null|EditorFileTab} currentButton
	 */
	static getNextAvailableButton(currentButton){
		let currentKey = null;

		if (currentButton !== null){
			currentKey = currentButton.filePath;
		}

		const keys = Object.keys(EditorFileTab.registeredButtons);

		// Find the numerical index of the current button key
		if (currentKey !== null){
			let indexOfCurrentKey = keys.indexOf(currentKey);
			if (indexOfCurrentKey > -1){
				if (indexOfCurrentKey > 0){
					return EditorFileTab.registeredButtons[keys[indexOfCurrentKey - 1]];
				}else{
					if (keys.length > 1) {
						return EditorFileTab.registeredButtons[keys[1]];
					}else{
						return null;
					}
				}
			}
		}else{
			// Get the next available key
			keys.reverse();
			if (keys.length > 0){
				return EditorFileTab.registeredButtons[keys[0]];
			}else{
				return null;
			}
		}
	}

	buttonsContainer;
	aceSession;
	/**
	 * @type {string} The full OS file-path of the file this instance represents
	 */
	filePath;
	/**
	 * @type {string} The file name, including extension, that this instance represents.
	 */
	fileName;
	/**
	 * @type {string} The last saved value of file contents for this instance.
	 */
	lastSavedValue;
	/**
	 * @type {HTMLButtonElement} The HTML element that this instance represents - the button in the editor tabs.
	 */
	dom;

	constructor(aceSession, filePath, fileName){
		this.buttonsContainer = document.querySelector("#content-editor-tabs-container");
		this.aceSession = aceSession;
		this.filePath = filePath;
		this.fileName = fileName;
		this.lastSavedValue = aceSession.getValue();
		this.dom = this.getDOM();

		this.dom.querySelector(".current-file-name-tab").textContent = fileName;

		// Connect event to show unsaved changes
		aceSession.on("change", () => {
			if (aceSession.getValue() !== this.lastSavedValue) {
				this.showUnsavedChanges();
			}else{
				this.hideUnsavedChanges();
			}
		});

		this.buttonsContainer.append(this.dom);
		EditorFileTab.registeredButtons[filePath] = this;
	}

	/**
	 * Builds an HTML element to represent the current instance.
	 * @returns {HTMLButtonElement}
	 */
	getDOM(){
		const button = document.createElement("button");
		button.setAttribute("type", "button");
		button.setAttribute("file-path", this.filePath);
		button.classList.add("editor-tab");
		button.innerHTML = `
			<span class="current-file-name-tab">Contents</span>
			<span style="display:none;" class="unsaved-changes">*</span>
			<div style="display:none;" class="saving-spinner spinner-border" role="status">
				<span class="sr-only">Loading...</span>
			</div>
			<span class="close-tab"><i class="bi bi-x-lg"></i></span>
		`;

		button.addEventListener("click", () => {
			this.select();
		});

		button.querySelector(".close-tab").addEventListener("click", async e => {
			e.stopPropagation();

			// Confirm this if there are unsaved changes
			let canClose = false;
			if (this.hasUnsavedChanges()){
				const confirmModal = new ConfirmModal("confirm-tab-closure-modal");
				confirmModal.setTitle("There are unsaved changes");
				confirmModal.setContent("You have unsaved changes on this tab. Do you want to close this tab without saving?");
				confirmModal.setConfirmButtonText("Close without saving");
				confirmModal.showModal();
				canClose = await confirmModal.actionTaken();
				confirmModal.cleanup();
			}else{
				canClose = true;
			}

			if (canClose) {
				this.aceSession.setValue("");

				if (EditorFileTab.getActiveTab() === this) {
					const nextButton = EditorFileTab.getNextAvailableButton(this);
					if (nextButton !== null) {
						nextButton.select();
					} else {
						// Close the editor I guess
						document.querySelector("#content-editor-container").style.display = "none";
						SaveFileButton.hide();
					}
				}

				this.dom.remove();
				delete EditorFileTab.registeredButtons[this.filePath];
			}
		});

		return button;
	}

	/**
	 * Fires the selection of this button, removing the selection from other buttons
	 * and setting this button's aceSession as the Ace Editor's current session.
	 *
	 * Logic for when a particular button is clicked is done here.
	 */
	select(){
		if (EditorFileTab.currentlySelectedButton !== this.dom){
			if (EditorFileTab.currentlySelectedButton !== null) {
				EditorFileTab.currentlySelectedButton.classList.remove("selected");
			}
			FileEditorStates.currentFilePath = this.filePath;
			EditorFileTab.currentlySelectedButton = this.dom;
			this.dom.classList.add("selected");
			const ace = DOMHelper.getThemeManagerAceEditors().file;
			ace.setSession(this.aceSession);

			// Check if the Layout Sections action button should be displayed.
			// This is checked by telling if the file path this button represents is a layout file
			if (FilePathUtility.isFileLayoutFile(this.filePath)){
				// Show the button, but also load the sections for this layout file
				document.querySelector("#layout-sections-button").style.display = null;
				FileEditor.layoutSectionsModal.modal.querySelector(".current-layout-file-name").textContent = FilePathUtility.getFileName(this.filePath);
				FileEditor.layoutSectionsModal.loadAndRenderFileSections(this.filePath);
			}else{
				// Hide the button
				document.querySelector("#layout-sections-button").style.display = "none";
			}
		}
	}

	hasUnsavedChanges(){
		const dom = this.dom;
		const unsavedChangesDom = dom.querySelector(".unsaved-changes");
		return unsavedChangesDom.style.display !== "none";
	}

	showUnsavedChanges(){
		const dom = this.dom;
		const unsavedChangesDom = dom.querySelector(".unsaved-changes");
		unsavedChangesDom.style.display = "inline";
	}

	hideUnsavedChanges(){
		const dom = this.dom;
		const unsavedChangesDom = dom.querySelector(".unsaved-changes");
		unsavedChangesDom.style.display = "none";
	}
}

export default EditorFileTab;