import Endpoints from "../../Endpoints.js";

// Modal singletons
import BlogReelModal from "../page-editor/modals/BlogReelModal.js";
import IPPReelModal from "../page-editor/modals/IPPReelModal.js";
import RecentArticlesModal from "../page-editor/modals/RecentArticlesModal.js";
import RecentProjectsModal from "../page-editor/modals/RecentProjectsModal.js";

// Forms
import BlogReelForm from "./forms/BlogReelForm.js";
import IPPReelForm from "./forms/IPPReelForm.js";
import RecentArticlesForm from "./forms/RecentArticlesForm.js";
import RecentProjectsForm from "./forms/RecentProjectsForm.js"

export default class EasyEditor {
	ActivePageId;
	IsProcessing = false;
	SaveButton = document.querySelector('.btn-success.btn-sm[type="submit"][form="page-edit-form"]');

	constructor(activePageId) {
		this.ActivePageId = activePageId;
		this.Initialize();
	}

	async Initialize() {
		const [phoneNumbers, faxNumbers] = await this.GetPhoneAndFaxNumbers();

		tinymce.init({
			selector: '#content-editor-container',
			width: '100%',
			promotion: false,
			statusbar: false,
			browser_spellcheck: true,
			plugins: 'table link lists',
			toolbar: 'undo redo | blocks | bold italic | alignleft aligncentre alignright alignjustify | indent outdent | bullist numlist',
			skin: 'tinymce-5',
			menu: {
				shortcodes: { title: 'Shortcodes', items: 'phoneNumbers faxNumbers companyName companyStreet companyPostal companyCity companyState blogReel ippReel recentArticles recentProjects reviews' }
			},
			menubar: 'edit view insert format shortcodes',
			setup: (editor) => {
				editor.ui.registry.addNestedMenuItem('phoneNumbers', {
					text: 'Phone Numbers',
					getSubmenuItems: () => {
						return phoneNumbers.map((phoneNumber, index) => ({
							type: 'menuitem',
							text: phoneNumber,
							onAction: () => editor.insertContent(this.GetPhoneNumbersShortcodeValue(index))
						}));
					}
				});
				editor.ui.registry.addNestedMenuItem('faxNumbers', {
					text: 'Fax Numbers',
					getSubmenuItems: () => {
						return faxNumbers.map((faxNumber, index) => ({
							type: 'menuitem',
							text: faxNumber,
							onAction: () => editor.insertContent(this.GetFaxNumbersShortcodeValue(index))
						}));
					}
				});

				new BlogReelForm(editor);
				new IPPReelForm(editor);
				new RecentArticlesForm(editor);
				new RecentProjectsForm(editor);

				const addMenuItem = (name, text, action) => {
					editor.ui.registry.addMenuItem(name, {
						text: text,
						onAction: action
					});
				};

				addMenuItem('companyName', 'Company Name', () => editor.insertContent(this.GetCompanyNameShortcodeValue()));
				addMenuItem('companyStreet', 'Company Street', () => editor.insertContent(this.GetCompanyStreetShortcodeValue()));
				addMenuItem('companyPostal', 'Company Postal', () => editor.insertContent(this.GetCompanyPostalShortcodeValue()));
				addMenuItem('companyCity', 'Company City', () => editor.insertContent(this.GetCompanyCityShortcodeValue()));
				addMenuItem('companyState', 'Company State', () => editor.insertContent(this.GetCompanyStateShortcodeValue()));
				addMenuItem('blogReel', 'Blog Reel', () => BlogReelModal.showModal());
				addMenuItem('ippReel', 'IPP Reel', () => IPPReelModal.showModal());
				addMenuItem('recentArticles', 'Recent Articles', () => RecentArticlesModal.showModal());
				addMenuItem('recentProjects', 'Recent Projects', () => RecentProjectsModal.showModal());
				addMenuItem('reviews', 'Reviews', () => editor.insertContent(this.GetReviewsShortcodeValue()));

				editor.addShortcut('ctrl+s', "Save page", () => {
					editor.save();
					this.SavePage();
				});
			}
		});

		// Handle saving
		this.SaveButton.addEventListener("click", () => {
			this.SavePage();
		});

		window.addEventListener("keydown", e => {
			if (e.ctrlKey && e.key === "s"){
				e.preventDefault();
				e.stopPropagation();
				this.SavePage();
			}
		});
	}

	async GetPhoneAndFaxNumbers(){
		const response = await fetch(`${Endpoints.easyEditor.getPhoneAndFaxNumbers}`, {
			cache:"no-cache",
			credentials:"same-origin",
			method:"get",
		});

		/** @type {{status: int, phoneNumbers: string[], faxNumbers}} */
		let data = await response.json();

		return [data.phoneNumbers, data.faxNumbers];
	}

	GetPhoneNumbersShortcodeValue(index) {
		// We add 1 to the index since starting at 0 seemed like it might confuse users
		return `{{ phone-number which='${index + 1}' }}`;
	}

	GetFaxNumbersShortcodeValue(index) {
		// We add 1 to the index since starting at 0 seemed like it might confuse users
		return `{{ fax-number which='${index + 1}' }}`;
	}

	GetCompanyNameShortcodeValue() {
		return '{{ company-name }}';
	}

	GetCompanyStreetShortcodeValue() {
		return '{{ company-street }}';
	}

	GetCompanyPostalShortcodeValue() {
		return '{{ company-postal }}';
	}

	GetCompanyCityShortcodeValue() {
		return '{{ company-city }}';
	}

	GetCompanyStateShortcodeValue() {
		return '{{ company-state }}';
	}

	GetReviewsShortcodeValue(){
		return '{{ get-reviews columns="1" truncate-body="false" limit="100" offset="0" city="" state="" schema="true" show-empty-message="true" show-header="true" }}';
	}

	async SavePage() {
		if (this.IsProcessing) {
			return;
		}

		this.IsProcessing = true;
		this.SaveButton.classList.add("disabled");

		const formData = new FormData();
		formData.set("content", tinymce.get('content-editor-container').getContent());

		const response = await fetch(`${Endpoints.easyEditor.savePage}/${this.ActivePageId}`, {
			body:formData,
			cache:"no-cache",
			credentials:"same-origin",
			method:"POST"
		});

		if(response.status !== 200){
			console.error("An error occurred while saving this page. Please contact a developer.");
		} else {
			console.log("success");
		}

		this.IsProcessing = false;
		this.SaveButton.classList.remove("disabled");
	}

	ChangePage(pageData){
		this.ActivePageId = pageData.id;

		tinymce.get('content-editor-container').setContent(pageData.pageBody);
	}
}