import Scheduler from "../../utils/Scheduler.js";
import ImageComponent from "../components/ImageComponent.js";

class ProjectMediaForm{
	currentProjectPageID = null;
	currentProjectImagesUploadDirectory = null;
	form = document.querySelector("#client-project-form-media-section");
	isProcessing = false;
	fileNamesThatFailedToUpload = [];

	/** @type {ProjectTextualForm} */
	textualForm;

	constructor(textualForm){
		this.textualForm = textualForm;
	}

	/**
	 * Submits each photo, including the cover photo, to the server to be validated before submitting the main form
	 * @return {Promise<boolean>} true if all image names are valid, and false if one failed.
	 */
	async validateAllImageFileNames(){
		// Validate the cover photo
		const mediaFData = new FormData(this.form);
		/** @type {File} */
		const coverPhotoFile = mediaFData.get("cover-photo");
		const coverPhotoForm = new FormData();
		coverPhotoForm.set("image", coverPhotoFile, coverPhotoFile.name);

		const response = await fetch(`/uplift/ipp-form/validate-image-file-name`, {
			body:coverPhotoForm,
			method:"POST",
			cache:"no-cache",
			credentials:"same-origin"
		});

		if (response.status === 200){
			let data;
			try{
				/** @type {{status: int, error: ?string}} **/
				data = await response.json();
			}catch(jsonSyntaxError) {
				const errorMessage = "The server sent back an error when attempting to validate image file names. Please send this message to your account representative.";
				this.textualForm.submittingError.showError(errorMessage);
				return false;
			}

			if (data.status === -1){
				const uploadError = data.error;
				this.textualForm.submittingError.showError(uploadError);
				return false;
			}
		}

		// If we're here, then we can validate each individual image component
		for (const component of ImageComponent.cache) {
			// Get the file, if it exists
			const fileInput = component.dom.querySelector(`input[type="file"]`);
			if (fileInput.files.length > 0) {
				/** @type {File} */
				const imageFile = fileInput.files[0];
				// Validate this image
				const imageFileForm = new FormData();
				imageFileForm.set("image", imageFile, imageFile.name);

				const response = await fetch(`/uplift/ipp-form/validate-image-file-name`, {
					body:imageFileForm,
					method:"POST",
					cache:"no-cache",
					credentials:"same-origin"
				});

				if (response.status === 200){
					let data;
					try{
						/** @type {{status: int, error: ?string}} **/
						data = await response.json();
					}catch(jsonSyntaxError) {
						const errorMessage = "The server sent back an error when attempting to validate image file names. Please send this message to your account representative.";
						this.textualForm.submittingError.showError(errorMessage);
						return false
					}

					if (data.status === -1){
						const uploadError = data.error;
						this.textualForm.submittingError.showError(uploadError);
						return false;
					}
				}
			}
		}

		return true;
	}

	/**
	 * This is for client-sided validation. The server will do server validation after everything is uploaded.
	 * Checks for at least a cover photo plus one additional image upload.
	 * @return {boolean} True if minimum amount of images are uploaded.
	 */
	doesHaveMinimumImagesSetToUpload(){
		const mediaFData = new FormData(this.form);

		/** @type {File} */
		const coverPhoto = mediaFData.get("cover-photo");
		if (coverPhoto.size > 0 && coverPhoto.name !== ""){
			for (const component of ImageComponent.cache) {
				// Get the file, if it exists
				const fileInput = component.dom.querySelector(`input[type="file"]`);
				if (fileInput.files.length > 0) {
					// Found at least one additional image. This, plus the cover photo requirement, has been met
					return true;
				}
			}
		}else{
			return false;
		}
	}

	/**
	 * Uploads the cover photo and all attached images for this project. Will return a list of the attached image file
	 * names (does not include the cover photo) to be sent to the final IPP API step in order to sort the gallery
	 * in the proper upload order. If null is returned, then the form was already in a submission process
	 * and refused to duplicate the action.
	 * @returns {Promise<string[] | null>}
	 */
	async onSubmit(){
		if (this.isProcessing){
			return null;
		}

		this.isProcessing = true;
		this.textualForm.submittingError.hide();
		this.textualForm.loader.setMediaLoaderAsInProgress();
		const mediaFData = new FormData(this.form);

		// Submit the cover photo
		/** @type {File} */
		const coverPhotoFile = mediaFData.get("cover-photo");
		const coverPhotoForm = new FormData();
		coverPhotoForm.set("image", coverPhotoFile, coverPhotoFile.name);
		coverPhotoForm.set("project-post-id", this.currentProjectPageID);
		coverPhotoForm.set("project-images-directory", this.currentProjectImagesUploadDirectory);

		const response = await fetch(`/uplift/ipp-form/cover-photo`, {
			body:coverPhotoForm,
			method:"POST",
			cache:"no-cache",
			credentials:"same-origin"
		});

		let data;
		// const responseClone = response.clone();
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			// const textualResponse = await responseClone.text();

			// Always shows this error because it is going to be the case 99% of the time
			// Servers don't allow PHP to catch the fatal memory error so we can't send a formatted message.
			// Production servers will always hide the error and log it instead, and then return a 500 error usually
			const errorMessage = "The cover photo provided is too large and caused the server to run out of memory when uploading and cropping. Please either resize the photo to be smaller or use an online image optimizer to get the file size lower in order to use that image as a cover photo.";
			this.textualForm.submittingError.showError(errorMessage);
			// SubmittingError.showError("There was an error uploading your cover photo. Your IPP was submitted, but none of the images were submitted. Please inform your account rep of this and attach the photos with the name of the project they should go to.");
			throw errorMessage;
		}

		if (data.status === 1){
			// Upload all the additional media

			/**
			 * We use this variable to keep track of the file names the upload-image backend
			 * API returns. The backend may slightly modify file names to be valid on the OS
			 * file system. This array will be sent to the finalization endpoint of the project API
			 * to allow for the gallery to be sorted in the correct order.
			 * @type {string[]}
			 */
			const listOfSuccessfullyUploadedFileNames = [];

			// Change on 12/20/2022 - Get the ImageComponents NOT from the cache
			// but from the DOM directly to preserve the drag'n'drop order
			for (const componentDOM of ImageComponent.componentsContainer.children){
				const component = ImageComponent.componentFromDOM(componentDOM);
				// Get the file, if it exists
				const fileInput = component.dom.querySelector(`input[type="file"]`);
				if (fileInput.files.length > 0){
					const file = fileInput.files[0];
					try {
						const uploadedFileName = await this.uploadImage(file);
						listOfSuccessfullyUploadedFileNames.push(uploadedFileName);
					}catch(err){
						this.fileNamesThatFailedToUpload.push(file.name);
					}
				}
			}

			// Minimum wait time
			await Scheduler.wait(850);
			this.isProcessing = false;
			return listOfSuccessfullyUploadedFileNames;
		}else if (data.status === -1){
			this.textualForm.submittingError.showError(data.error);
			this.isProcessing = false;
			throw data.error;
		}
	}

	/**
	 * @param {File} file
	 * @returns {Promise<string>} The file name that successfully was uploaded
	 * @throws
	 */
	async uploadImage(file){

		const fData = new FormData();
		fData.set("image", file, file.name);
		fData.set("project-images-directory", this.currentProjectImagesUploadDirectory);

		const response = await fetch(`/uplift/ipp-form/image`, {
			body:fData,
			method:"POST",
			cache:"no-cache",
			credentials:"same-origin"
		});

		let data;
		try{
			/** @type {{status: int, error: ?string, fileName: string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			const errorMessage = `Your image file ${file.name} failed to upload due to an error with the image. We cannot create your project post at this time. Please report this exact error (Screenshot it) to our team and we will investigate it. Additionally, provide the images you tried to upload.`;
			this.textualForm.submittingError.showError(errorMessage);
			throw errorMessage;
		}

		if (data.status === 1){
			// The API will return the uploaded image file name
			// This is important because sometimes the file name will be modified to be
			// valid in the OS file system.
			return data.fileName;
		}else if (data.status === -1){
			throw data.error;
		}
	}
}

export default ProjectMediaForm;