class ImageComponent{

	/** @type {ImageComponent[]} */
	static cache = [];
	static internalCounter = 0;
	static componentsContainer = document.querySelector("#additional-image-components-container");
	/** @type {?ImageComponent} */
	static currentDraggingInstance = null;

	/** @type {?HTMLDivElement} */
	dom = null;
	/** @type {?HTMLDivElement} */
	uploadContainerDOM = null;
	isDragging = false;
	currentDraggingYOffsetFromMouse = 0;

	static{
		window.addEventListener("mousemove", /** @param {MouseEvent} e */ e => {
			if (ImageComponent.currentDraggingInstance !== null){
				ImageComponent.currentDraggingInstance.dragMoved(e.pageY);
			}
		});

		window.addEventListener("touchmove", /** @param {TouchEvent} e */ e => {
			if (ImageComponent.currentDraggingInstance !== null){
				/** @type {Touch} */
				const firstTouch = e.touches[0];
				if (firstTouch) {
					e.preventDefault();
					ImageComponent.currentDraggingInstance.dragMoved(firstTouch.pageY);
				}
			}
		}, {passive: false});

		window.addEventListener("mouseup", /** @param {MouseEvent} e */ e => {
			if (ImageComponent.currentDraggingInstance !== null){
				ImageComponent.currentDraggingInstance.dragEnded();
			}
		});

		window.addEventListener("touchend", /** @param {TouchEvent} e */ e => {
			if (ImageComponent.currentDraggingInstance !== null){
				ImageComponent.currentDraggingInstance.dragEnded();
			}
		}, {passive: false});

		window.addEventListener("touchcancel", /** @param {TouchEvent} e */ e => {
			if (ImageComponent.currentDraggingInstance !== null){
				ImageComponent.currentDraggingInstance.dragEnded();
			}
		}, {passive: false});
	}

	/**
	 * Gets an ImageComponent from an HTMLDivElement
	 * @param {HTMLDivElement} dom
	 */
	static componentFromDOM(dom){
		for (const component of ImageComponent.cache){
			if (component.dom === dom){
				return component;
			}
		}

		return null;
	}

	static removeFromCacheByID(id){
		for (const componentIndex in ImageComponent.cache){
			const component = ImageComponent.cache[parseInt(componentIndex)];
			if (component.id === id){
				ImageComponent.cache.splice(parseInt(componentIndex), 1);
				break;
			}
		}
	}

	constructor(){
		this.id = ++ImageComponent.internalCounter;
		this.dom = this.getDOM();
		this.uploadContainerDOM = this.dom.querySelector(".client-project-form-fake-upload-container");

		ImageComponent.cache.push(this);
	}

	remove(){
		ImageComponent.removeFromCacheByID(this.id);
		this.dom.remove();
	}

	getDOM(){
		const template = document.createElement("div");
		template.classList.add("client-project-form-upload-wrapper");
		template.innerHTML = `
			<div class="client-project-form-fake-upload-container">
				<div class="move-handles-container">
					<button type="button" class="move-handle">
						<i class="bi bi-arrows-move"></i>
					</button>
				</div>
				<div class="input-container">
					<input accept=".jpeg,.jpg,.png,.gif,.webp,.heic" type="file" id="additional-image-${this.id}">
					<label for="additional-image-${this.id}"></label>
				</div>
				<div class="buttons-container">
					<button type="button" class="remove-button btn btn-sm btn-danger">
						<i class="bi bi-trash-fill"></i>
						<span>Remove Image</span>
					</button>
				</div>
			</div>
		`;

		const dragHandle = template.querySelector(".move-handle");
		const removeButton = template.querySelector(".remove-button");
		const fileInput = template.querySelector(".input-container input");

		removeButton.addEventListener("click", () => {
			this.remove();
		});

		fileInput.addEventListener("change", /** @param {InputEvent} e */ e => {
			this.onFileInputChanged(fileInput, e);
		});

		dragHandle.addEventListener("mousedown",  /** @param {MouseEvent} e */ e => {
			this.startDrag(e.pageY);
		});

		dragHandle.addEventListener("touchstart",  /** @param {TouchEvent} e */ e => {
			/** @type {Touch} */
			const firstTouch = e.touches[0];
			if (firstTouch) {
				this.startDrag(firstTouch.pageY);
			}
		}, {passive: false});

		ImageComponent.componentsContainer.prepend(template);

		return template;
	}

	/**
	 * @param {HTMLInputElement} inputElement
	 * @param {InputEvent} event
	 */
	onFileInputChanged(inputElement, event){
		const file = inputElement.files[0];
		const maxFileSizeInBytes = parseInt(document.querySelector("#max-file-upload-size").getAttribute("data-in-bytes"));
		if (file){
			// Don't let files be 95% max file size
			const paddedMaxFileSizeInBytes = 0.95 * maxFileSizeInBytes;
			const fileSize = file.size;
			if (fileSize >= paddedMaxFileSizeInBytes){
				alert("That file is too large and cannot be submitted. We have removed it. Please optimize or resize that file to submit it.");
				inputElement.value = null;
			}else{
				// It's good
			}
		}
	}

	startDrag(pageY){
		const boundingRectsOfDOM = this.uploadContainerDOM.getBoundingClientRect();
		const boundingRectsOfContainer = ImageComponent.componentsContainer.getBoundingClientRect();
		const leftOffset = boundingRectsOfContainer.left;

		// How many pixels is the mouse inside the element that is being dragged?
		// Use this to offset the dragging operation to avoid the element snapping to the top edge of itself
		// when being dragged. Allows the use to drag from wherever their mouse/finger touched
		this.currentDraggingYOffsetFromMouse = pageY - (boundingRectsOfDOM.top + window.scrollY);

		this.isDragging = true;
		this.dom.style.height = `${this.uploadContainerDOM.clientHeight}px`;
		document.body.append(this.uploadContainerDOM);
		this.uploadContainerDOM.classList.add("dragging");
		ImageComponent.currentDraggingInstance = this;
		this.uploadContainerDOM.style.top = `${pageY - this.currentDraggingYOffsetFromMouse}px`;
		this.uploadContainerDOM.style.left = `${leftOffset}px`;
	}

	dragMoved(pageY){
		this.uploadContainerDOM.style.top = `${pageY - this.currentDraggingYOffsetFromMouse}px`;

		// Check previous sibling and next sibling

		if (this.dom.previousElementSibling){
			const prevElementSiblingYPosition = (this.dom.previousElementSibling.getBoundingClientRect().top + window.scrollY);
			if (pageY < prevElementSiblingYPosition){
				// Swap positions
				this.dom.parentElement.insertBefore(this.dom, this.dom.previousElementSibling);
			}
		}

		if (this.dom.nextElementSibling){
			const nextElementSiblingYPosition = (this.dom.nextElementSibling.getBoundingClientRect().top + window.scrollY);
			if (pageY > nextElementSiblingYPosition){
				// Swap positions
				this.dom.parentElement.insertBefore(this.dom.nextElementSibling, this.dom);
			}
		}

	}

	dragEnded(){
		this.isDragging = false;
		this.dom.style.height = null;
		this.dom.append(this.uploadContainerDOM);
		this.uploadContainerDOM.classList.remove("dragging");
		ImageComponent.currentDraggingInstance = null;
	}
}

export default ImageComponent;