import ActivityLog from "./components/ActivityLog.js";
import Endpoints from "../Endpoints.js";
import Paginator from "../utils/Paginator.js";
import Scheduler from "../utils/Scheduler.js";

class ActivityLogs
{
	currentPage = 1;
	currentLimit = 25;
	currentCategory = 0;
	currentUser = document.querySelector("#activity-log-rows-container").dataset.user;
	loader = document.querySelector("#activity-log-loader");
	categorySelect = document.querySelector("#category-select");
	isProcessing = false;

	/**
	 * @type {Paginator[]}
	 */
	paginators = [];

	constructor()
	{
		for (const paginator of document.querySelectorAll("pagination")){
			const newPaginator = new Paginator(paginator);
			this.hookPaginatorEvents(newPaginator);
			this.paginators.push(newPaginator);
		}

		this.hookCategoryEvent();

		this.loadActivityLogs(
			this.currentPage,
			this.currentLimit,
			this.currentCategory,
			this.currentUser,
		);
	}

	/**
	 * @param {Paginator} paginator
	 */
	hookPaginatorEvents(paginator){
		paginator.onPagePrev(pageNumber => {
			this.currentPage = pageNumber;
			this.loadActivityLogs(
				pageNumber,
				this.currentLimit,
				this.currentCategory,
				this.currentUser,
			)
		});

		paginator.onPageNext(pageNumber => {
			this.currentPage = pageNumber;
			this.loadActivityLogs(
				pageNumber,
				this.currentLimit,
				this.currentCategory,
				this.currentUser,
			)
		});

		paginator.onPageManuallyEntered(pageNumber => {
			this.currentPage = pageNumber;
			this.loadActivityLogs(
				pageNumber,
				this.currentLimit,
				this.currentCategory,
				this.currentUser,
			)
		});
	}

	hookCategoryEvent() {
		this.categorySelect.addEventListener("change", e => {
			document.querySelector("input[name='paginator-current-page']").value = 1;
			this.loadActivityLogs(
				1,
				this.currentLimit,
				this.categorySelect.value,
				this.currentUser,
			);
		});
	}

	/**
	 * @param {int} page
	 * @param {int} limit
	 * @param {int} category
	 * @param {string} user
	 * @returns {Promise<void>}
	 */
	async loadActivityLogs(page, limit, category, user)
	{
		if (this.isProcessing){
			return;
		}

		ActivityLog.activityLogsContainer.innerHTML = "";

		this.isProcessing = true;
		this.loader.style.display = "block";
		this.currentPage = page;
		this.currentLimit = limit;
		this.currentCategory = category;
		this.currentUser = user;

		const urlParams = new URLSearchParams();
		urlParams.set("limit", String(limit));
		urlParams.set("page", String(page));
		urlParams.set("category", String(category));
		urlParams.set("user", user);

		const response = await fetch(`${Endpoints.activityLogs}?${urlParams.toString()}`, {
			cache:"no-cache",
			credentials:"same-origin",
			method:"get",
		});

		await Scheduler.wait(150);

		let data;
		try{
			data = await response.json();
		}catch(jsonSyntaxError){
			alert(jsonSyntaxError);
			return;
		}

		this.loader.style.display = "none";

		if (data.status === 1){
			// Set paginator stuff
			for (const paginator of this.paginators){
				paginator.setMaxPages(data.totalPages);
			}

			for (const activityLog of data.activityLogs){
				new ActivityLog(
					activityLog.id,
					activityLog.categoryID,
					activityLog.userID,
					activityLog.timestamp,
					activityLog.jsonData,
					activityLog.username,
					activityLog.categoryDescription,
					activityLog.timeAgo,
				);
			}

			if(data.activityLogs.length === 0) {
				document.querySelector("#no-logs-to-display-message").style.display = "block";
			} else {
				document.querySelector("#no-logs-to-display-message").style.display = "none";
			}
		}else if (data.status === -1){
			alert(data.error);
		}

		this.isProcessing = false;
	}
}

export default new ActivityLogs();