<?php

	namespace Accounts;

	use PHPUnit\Framework\TestCase;
	use Roles\Permission;
	use Roles\PermissionCategories;
	use Roles\Role;

	final class AccountTests extends TestCase
	{

		private static Account $account;
		private static AccountCookieSession $token;
		private static Role $role;
		private static Permission $permission;

		public static function setUpBeforeClass(): void
		{
			$_COOKIE = [];

			$role = new Role();
			$role->name = "Test Role";
			$role->save();
			self::$role = $role;

			$permission = new Permission();
			$permission->permissionID = PermissionCategories::EDIT_PAGES->value;
			$permission->roleID = $role->id;
			$permission->isEnabled = 1;
			$permission->save();
			self::$permission = $permission;

			$account = new Account();
			$account->firstName = "Nox";
			$account->lastName = "7";
			$account->username = "nox7";
			$account->roleID = $role->id;
			$account->save();
			self::$account = $account;

			$accountToken = new AccountCookieSession();
			$accountToken->userID = $account->id;
			$accountToken->expires = time() + 100000;
			$accountToken->cookie = "TestCookie";
			$accountToken->save();
			self::$token = $accountToken;

			$_COOKIE[Account::LOGIN_TOKEN_NAME] = $accountToken->cookie;
		}

		public static function tearDownAfterClass(): void
		{
			self::$role->delete();
			self::$permission->delete();
			self::$account->delete();
			self::$token->delete();
		}

		/**
		 * @return void
		 */
		public function testGetCurrentUser(): void{
			$newAdminAccount = Account::getCurrentUser();
			$this->assertInstanceOf(
				expected: Account::class,
				actual: $newAdminAccount,
			);
		}

		public function testGetAccountFullName(): void{
			$this->assertEquals(
				expected: "Nox 7",
				actual: self::$account->getFullName(),
			);
		}

		public function testAccountHasPermission(): void{
			$this->assertEquals(
				expected: true,
				actual: self::$account->hasPermission(PermissionCategories::EDIT_PAGES)
			);
		}

		public function testAccountHasAnyPermission(): void{
			$this->assertEquals(
				expected: true,
				actual: self::$account->hasAnyPermission([PermissionCategories::EDIT_PAGES])
			);
		}

		public function testAccountHasAnyPermissionWithMultiplePermissionsInArray(): void{
			$this->assertEquals(
				expected: true,
				actual: self::$account->hasAnyPermission([PermissionCategories::EDIT_PAGES, PermissionCategories::MANAGE_ROLES])
			);
		}

		public function testAccountDoesNotHavePermission(): void{
			$this->assertEquals(
				expected: false,
				actual: self::$account->hasPermission(PermissionCategories::MANAGE_ROLES)
			);
		}
	}